package com.atlassian.bamboo.stash;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.ApplicationLinkRequest;
import com.atlassian.applinks.api.ApplicationLinkRequestFactory;
import com.atlassian.applinks.api.ApplicationLinkService;
import com.atlassian.applinks.api.CredentialsRequiredException;
import com.atlassian.applinks.api.application.bitbucket.BitbucketApplicationType;
import com.atlassian.applinks.api.auth.types.TwoLeggedOAuthAuthenticationProvider;
import com.atlassian.applinks.spi.util.TypeAccessor;
import com.atlassian.bamboo.applinks.CredentialsRequiredContextException;
import com.atlassian.bamboo.chains.ChainResultsSummary;
import com.atlassian.bamboo.chains.branches.MergeResultState;
import com.atlassian.bamboo.chains.branches.MergeResultSummary;
import com.atlassian.bamboo.configuration.AdministrationConfigurationAccessor;
import com.atlassian.bamboo.jira.rest.Errors;
import com.atlassian.bamboo.jira.rest.JiraRestResponse;
import com.atlassian.bamboo.logger.ErrorHandler;
import com.atlassian.bamboo.notification.Notification;
import com.atlassian.bamboo.notification.NotificationRecipient;
import com.atlassian.bamboo.notification.NotificationTransport;
import com.atlassian.bamboo.notification.recipients.AbstractNotificationRecipient;
import com.atlassian.bamboo.plan.PlanResultKey;
import com.atlassian.bamboo.plugin.descriptor.NotificationRecipientModuleDescriptor;
import com.atlassian.bamboo.resultsummary.ResultsSummary;
import com.atlassian.bamboo.resultsummary.vcs.RepositoryChangeset;
import com.atlassian.bamboo.spring.ComponentAccessor;
import com.atlassian.bamboo.template.TemplateRenderer;
import com.atlassian.bamboo.util.Narrow;
import com.atlassian.bamboo.utils.BambooUrl;
import com.atlassian.bamboo.utils.error.ErrorCollection;
import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import com.atlassian.sal.api.net.ResponseTimeoutException;
import com.atlassian.sal.api.net.ReturningResponseHandler;
import com.atlassian.struts.TextProvider;
import com.google.common.base.Supplier;
import com.opensymphony.webwork.dispatcher.json.JSONException;
import com.opensymphony.webwork.dispatcher.json.JSONObject;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.atlassian.sal.api.net.Request.MethodType.POST;

public class StashNotificationRecipient extends AbstractNotificationRecipient implements NotificationRecipient.RequiresResultSummary
{
    private static final Logger log = Logger.getLogger(StashNotificationRecipient.class);

    private final String STASH_REST_URL_POINT = "rest/build-status/latest/commits/";
    private final String NOTIFICATION_BUILD_STATUS = "com/atlassian/bamboo/stash/notificationBuildStatus.ftl";
    private final String CONDITION_KEY = "conditionKey";


    private ResultsSummary resultsSummary;

    private TemplateRenderer templateRenderer;
    private TextProvider textProvider;
    private ErrorHandler errorHandler;
    private Supplier<ApplicationLinkService> applicationLinkService = ComponentAccessor.newLazyComponentReference("applinkApplicationLink");
    private Supplier<TypeAccessor> typeAccessor = ComponentAccessor.newLazyComponentReference("applinkTypeAccessor");
    private AdministrationConfigurationAccessor administrationConfigurationAccessor;

    @NotNull
    @Override
    public String getEditHtml()
    {
        String editTemplateLocation = ((NotificationRecipientModuleDescriptor)getModuleDescriptor()).getEditTemplate();
        return templateRenderer.render(editTemplateLocation, getRenderContextForConfig());
    }

    @NotNull
    @Override
    public String getViewHtml()
    {
        String viewTemplateLocation = ((NotificationRecipientModuleDescriptor)getModuleDescriptor()).getViewTemplate();
        return templateRenderer.render(viewTemplateLocation, getRenderContextForConfig());
    }

    @NotNull
    @Override
    public ErrorCollection validate(@NotNull Map<String, String[]> params)
    {
        ErrorCollection errorCollection = super.validate(params);
        String conditionKey = params.containsKey(CONDITION_KEY) ? params.get(CONDITION_KEY)[0] : null;
        if (!"com.atlassian.bamboo.plugin.system.notifications:chainCompleted.allBuilds".equals(conditionKey))
        {
            errorCollection.addError(CONDITION_KEY, textProvider.getText("bamboo.stash.plugin.invalid.event.type"));
        }
        return errorCollection;
    }

    @NotNull
    @Override
    public List<NotificationTransport> getTransports()
    {
        List<NotificationTransport> list = new ArrayList<>();
        list.add(new NotificationTransport()
        {
            @Override
            public void sendNotification(@NotNull Notification notification)
            {
                String message = StringUtils.defaultString(templateRenderer.render(NOTIFICATION_BUILD_STATUS, getRenderContextForMessage()));
                //String message = StringUtils.defaultString(notification.getIMContent());

                PlanResultKey planResultKey = resultsSummary.getPlanResultKey();

                JSONObject jsonData;
                try
                {
                    // https://extranet.atlassian.com/display/STASH/Build+Status#BuildStatus-RESTResource
                    jsonData = new JSONObject()
                            .put("state", resultsSummary.isSuccessful() ? "SUCCESSFUL" : "FAILED")
                            .put("key", planResultKey.getPlanKey())
                            .put("name", resultsSummary.getImmutablePlan().getName())
                            .put("url", new BambooUrl(administrationConfigurationAccessor).withBaseUrlFromConfiguration("/browse/" + planResultKey))
                            .put("description", message.length() > 255 ? message.substring(0, 255) : message)
                    ;
                }
                catch (JSONException e)
                {
                    //review: what should else I do?
                    throw new RuntimeException(e);
                }

                boolean hasSentAtLeastOneBuildStatus = false;
                for(ApplicationLink applicationLink : applicationLinkService.get().getApplicationLinks(typeAccessor.get().getApplicationType(BitbucketApplicationType.class).getClass()))
                {
                    for (RepositoryChangeset repositoryChangeset : resultsSummary.getRepositoryChangesets())
                    {
                        String changesetId = repositoryChangeset.getChangesetId();
                        hasSentAtLeastOneBuildStatus |= sendStashNotification(planResultKey, jsonData, applicationLink, changesetId);
                    }
                    ChainResultsSummary chainResultsSummary = Narrow.to(resultsSummary, ChainResultsSummary.class);
                    if (chainResultsSummary != null)
                    {
                        MergeResultSummary mergeResult = chainResultsSummary.getMergeResult();
                        if (mergeResult != null)
                        {
                            if (mergeResult.getPushState() == MergeResultState.SUCCESS)
                            {
                                String mergeResultVcsKey = mergeResult.getMergeResultVcsKey();
                                hasSentAtLeastOneBuildStatus |= sendStashNotification(planResultKey, jsonData, applicationLink, mergeResultVcsKey);
                            }
                        }
                    }
                }
                if (!hasSentAtLeastOneBuildStatus)
                {
                    errorHandler.recordError(planResultKey, "No build status has been sent even though there is a Stash notification configured for that plan. "
                            + "Is your application link configured properly?", null);
                }
            }
        });
        return list;
    }

    private boolean sendStashNotification(final PlanResultKey planResultKey,
                                          final JSONObject jsonData,
                                          final ApplicationLink applicationLink,
                                          final String changesetId)
    {
        if (changesetId.length() != 40) //it's not the git 40char hash, no need to spam Stash
        {
            return false;
        }
        try
        {
            // see comment below
            //  JiraRestResponse response = jiraRestService.get().doRestCallViaApplink(applicationLink, STASH_REST_URL_POINT + changesetId, POST, jsonData, BasicAuthenticationProvider.class);
            JiraRestResponse response = doRestCallViaApplink(applicationLink, STASH_REST_URL_POINT + changesetId, POST, jsonData);
            if (response.hasErrors() || response.statusCode != HttpStatus.SC_NO_CONTENT)
            {
                log.error("Failed to store build status against Stash instance " + applicationLink.getName() + ", reason: " + response);
                if (response.statusCode == HttpStatus.SC_UNAUTHORIZED)
                {
                    errorHandler.recordError(planResultKey, "\"401 Unauthorized\" received while attempting to store build status against Stash instance `"
                                                            + applicationLink.getName() + "', check your application link configuration.", null);
                }
            }
            else
            {
                return true;
            }
        }
        catch (CredentialsRequiredContextException e)
        {
            //uga buga? what else we should/could do here?
            log.error("Credentials required while attempting to store build status against Stash instance " + applicationLink.getName());
        }
        return false;
    }

    // this is copy paste of JiraRestServiceImpl.doRestCallViaApplink. The only real difference is that this one method allows to specify which
    // authenticationProvider to use (in bamboo-stash-plugin case: TwoLeggedOAuth). The rest of logic is the same. This should be later (Bamboo 5.0) replaced with a call
    // to extended JiraRestService allowing to pass desired authProvider as an additional param, but currently I want to be backward compatible)
    public JiraRestResponse doRestCallViaApplink(@NotNull ApplicationLink applicationLink, @NotNull String requestUrl, @NotNull Request.MethodType methodType, @Nullable JSONObject data) throws CredentialsRequiredContextException
    {
        final ApplicationLinkRequestFactory requestFactory = applicationLink.createAuthenticatedRequestFactory(TwoLeggedOAuthAuthenticationProvider.class); // <-- here, the difference
        if (requestFactory == null)
        {
            String errorMessage = "Failed to send stash build notification via " + applicationLink.getDisplayUrl() + " - it has to have OAuth authentication configured.";
            log.warn(errorMessage);
            return new JiraRestResponse(errorMessage);
        }
        try
        {
            if (log.isDebugEnabled())
            {
                log.debug("Executing JIRA request. " + methodType + " " + applicationLink.getDisplayUrl() + "/" + requestUrl);
            }

            final ApplicationLinkRequest request = requestFactory.createRequest(methodType, requestUrl);
            request.setHeader(HttpHeaders.CONTENT_TYPE, "application/json");
            if (methodType != Request.MethodType.GET && data != null)
            {
                request.setRequestBody(data.toString());
            }
            request.setSoTimeout((int) TimeUnit.SECONDS.toMillis(30));

            JiraRestResponse response = request.executeAndReturn(new JiraResponseHandler());
//            response = validateJiraNotBelowFiveZero(response, applicationLink, requestFactory); // not needed for stash
            if (response.hasErrors())
            {
                log.warn("Failed to execute application link request. Server: " + applicationLink.getDisplayUrl() + " Method: " + methodType + " Url: " + requestUrl + "\n" +
                        "Response: " + response.statusCode + " " + response.statusMessage + "\n" +
                        "Errors: " + response.errors.toString());

                if (log.isDebugEnabled() && StringUtils.isNotBlank(response.body))
                {
                    log.debug("Response body: " + response.body);
                }
            }
            return response;
        }
        catch (ResponseTimeoutException e)
        {
            log.warn("Request to JIRA timed out. Server: " + applicationLink.getDisplayUrl() + " Method: " + methodType + " Url: " + requestUrl);
            log.warn(e.getMessage(), e);
            return new JiraRestResponse("Request to JIRA timed out: " + e.getMessage());
        }
        catch (ResponseException e)
        {
            log.warn("Failed to execute JIRA request. Server: " + applicationLink.getDisplayUrl() + " Method: " + methodType + " Url: " + requestUrl);
            log.warn(e.getMessage(), e);
            return new JiraRestResponse("Request to JIRA failed: " + e.getMessage());
        }
        catch (CredentialsRequiredException e)
        {
            log.info("Authentication was required, but credentials were not available when doing applink call. "
                    + "Server: " + applicationLink.getDisplayUrl() + " Method: " + methodType + " Url: " + requestUrl);
            throw new CredentialsRequiredContextException(applicationLink.getName(), e);
        }
    }

    /* utility class to execute the applink request and returning response as String */
    static class JiraResponseHandler implements ReturningResponseHandler<Response, JiraRestResponse>
    {
        public JiraRestResponse handle(Response response)
        {
            JiraRestResponse.JiraRestResponseBuilder respBuilder = new JiraRestResponse.JiraRestResponseBuilder(response.getStatusCode(), response.getStatusText());
            if (!response.isSuccessful())
            {
                try
                {
                    //standard JIRA Error response
                    respBuilder.errors(response.getEntity(Errors.class));
                }
                catch (Exception entityException)
                {
                    try
                    {
                        //non standard error response
                        String responseString = IOUtils.toString(response.getResponseBodyAsStream());
                        if (StringUtils.isNotBlank(responseString))
                        {
                            respBuilder.body(responseString);
                            respBuilder.addError("Request to JIRA failed. Returned with " + response.getStatusCode() + ". Response: " + responseString);
                        }
                    }
                    catch (Exception bodyException)
                    {
                        // no error details
                        respBuilder.addError("Request to JIRA failed. Returned with " + response.getStatusCode());
                    }
                }
            }
            else
            {
                try
                {
                    String responseString = IOUtils.toString(response.getResponseBodyAsStream());
                    respBuilder.body(responseString);

                    if (StringUtils.isNotBlank(responseString))
                    {
                        try
                        {
                            JSONObject jsonObject = new JSONObject(responseString);
                            respBuilder.entity(jsonObject);
                        }
                        catch (JSONException e)
                        {
                            respBuilder.addError("Failed to parse response: " + e.getMessage());
                        }
                    }
                }
                catch (Exception e)
                {
                    log.debug("Failed to extract body of JIRA's response", e);
                }
            }

            return respBuilder.build();
        }
    }
    //////////////////////////// end f copy paste from JiraRestServiceImpl.java ////////////////////////////////////////

    private Map<String, Object> getRenderContextForMessage()
    {
        Map<String, Object> context = new HashMap<>();
        context.put("resultsSummary", resultsSummary);
        context.put("buildSummary", resultsSummary); // must be buildSummary because @nc.showTestSummary macro have that variable hard-coded :(
        return context;
    }

    private Map<String, Object> getRenderContextForConfig()
    {
        Map<String, Object> context = new HashMap<>();
        boolean hasStashApplinkConfiguredProperly = false;
        for (ApplicationLink stashApplicationLink : applicationLinkService.get().getApplicationLinks(typeAccessor.get().getApplicationType(BitbucketApplicationType.class).getClass()))
        {
            ApplicationLinkRequestFactory requestFactory = stashApplicationLink.createAuthenticatedRequestFactory(TwoLeggedOAuthAuthenticationProvider.class);
            if (requestFactory != null)
            {
                hasStashApplinkConfiguredProperly = true;
                break;
            }
        }
        if (!hasStashApplinkConfiguredProperly)
        {
            context.put("noStashApplinkConfigured", true);
        }
        return context;
    }

    @Override
    public void setResultsSummary(@Nullable final ResultsSummary resultsSummary)
    {
        this.resultsSummary = resultsSummary;
    }

    //-----------------------------------Dependencies
    public void setTemplateRenderer(TemplateRenderer templateRenderer)
    {
        this.templateRenderer = templateRenderer;
    }

    public void setTextProvider(TextProvider textProvider)
    {
        this.textProvider = textProvider;
    }

    public void setErrorHandler(ErrorHandler errorHandler)
    {
        this.errorHandler = errorHandler;
    }

    public void setAdministrationConfigurationAccessor(AdministrationConfigurationAccessor administrationConfigurationAccessor)
    {
        this.administrationConfigurationAccessor = administrationConfigurationAccessor;
    }
}
