package com.atlassian.bamboo.plugins.nodejs.tasks.mocha.parser;

import com.atlassian.bamboo.specs.api.model.task.TaskProperties;
import com.atlassian.bamboo.specs.api.validators.common.ValidationContext;
import com.atlassian.bamboo.specs.api.validators.common.ValidationProblem;
import com.atlassian.bamboo.specs.builders.task.MochaParserTask;
import com.atlassian.bamboo.specs.model.task.MochaParserTaskProperties;
import com.atlassian.bamboo.task.TaskConfigConstants;
import com.atlassian.bamboo.task.TaskContainer;
import com.atlassian.bamboo.task.TaskDefinition;
import com.atlassian.bamboo.task.export.TaskDefinitionExporter;
import com.atlassian.bamboo.task.export.TaskValidationContext;
import com.atlassian.bamboo.util.Narrow;
import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class MochaParserTaskExporter implements TaskDefinitionExporter {
    @NotNull
    private ValidationContext getValidationContext() {
        return MochaParserTaskProperties.VALIDATION_CONTEXT;
    }

    @NotNull
    @Override
    public Map<String, String> toTaskConfiguration(@NotNull TaskContainer taskContainer, @NotNull TaskProperties taskProperties) {
        final MochaParserTaskProperties typedTaskProperties = Narrow.downTo(taskProperties, MochaParserTaskProperties.class);
        Preconditions.checkState(typedTaskProperties != null, "Don't know how to import task properties of type: " + taskProperties.getClass().getName());

        final Map<String, String> config = new HashMap<>();
        config.put(MochaParserConfigurator.PATTERN, typedTaskProperties.getTestFilePattern());
        config.put(TaskConfigConstants.CFG_WORKING_SUBDIRECTORY, typedTaskProperties.getWorkingSubdirectory());
        config.put(TaskConfigConstants.CFG_TEST_OUTDATED_RESULTS_FILE, Boolean.toString(typedTaskProperties.isPickUpTestResultsCreatedOutsideOfThisBuild()));

        return config;
    }

    @NotNull
    @Override
    public MochaParserTask toSpecsEntity(@NotNull TaskDefinition taskDefinition) {
        final Map<String, String> configuration = taskDefinition.getConfiguration();
        return new MochaParserTask()
                .testFilePattern(configuration.get(MochaParserConfigurator.PATTERN))
                .workingSubdirectory(configuration.getOrDefault(TaskConfigConstants.CFG_WORKING_SUBDIRECTORY, null))
                .pickUpTestResultsCreatedOutsideOfThisBuild(Boolean.parseBoolean(configuration.getOrDefault(TaskConfigConstants.CFG_TEST_OUTDATED_RESULTS_FILE, null)));
    }

    @Override
    public List<ValidationProblem> validate(@NotNull TaskValidationContext taskValidationContext, @NotNull TaskProperties taskProperties) {
        final List<ValidationProblem> validationProblems = new ArrayList<>();

        final MochaParserTaskProperties typedTaskProperties = Narrow.downTo(taskProperties, MochaParserTaskProperties.class);
        if (typedTaskProperties != null) {
            if (StringUtils.isBlank(typedTaskProperties.getTestFilePattern())) {
                validationProblems.add(new ValidationProblem(getValidationContext(), "Test file pattern is not defined"));
            }
        }

        return validationProblems;
    }
}
