/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bamboo.plugins.git;

import com.atlassian.bamboo.build.logger.BuildLogger;
import com.atlassian.bamboo.commit.Commit;
import com.atlassian.bamboo.commit.CommitContext;
import com.atlassian.bamboo.plugins.git.AncestryPathOutputHandler;
import com.atlassian.bamboo.plugins.git.CommitOutputHandler;
import com.atlassian.bamboo.plugins.git.GitCommandBuilder;
import com.atlassian.bamboo.plugins.git.GitCommandException;
import com.atlassian.bamboo.plugins.git.GitRepositoryAccessData;
import com.atlassian.bamboo.process.BambooProcessHandler;
import com.atlassian.bamboo.repository.RepositoryException;
import com.atlassian.bamboo.ssh.ProxyErrorReceiver;
import com.atlassian.bamboo.util.BambooFilenameUtils;
import com.atlassian.bamboo.util.BambooStringUtils;
import com.atlassian.bamboo.util.Narrow;
import com.atlassian.bamboo.util.PasswordMaskingUtils;
import com.atlassian.bamboo.util.SharedTemporaryFiles;
import com.atlassian.bamboo.utils.BambooPredicates;
import com.atlassian.bamboo.utils.Pair;
import com.atlassian.utils.process.ExternalProcess;
import com.atlassian.utils.process.ExternalProcessBuilder;
import com.atlassian.utils.process.LineOutputHandler;
import com.atlassian.utils.process.OutputHandler;
import com.atlassian.utils.process.ProcessHandler;
import com.atlassian.utils.process.StringOutputHandler;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.SystemUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

class GitCommandProcessor
implements Serializable,
ProxyErrorReceiver {
    private static final Logger log = Logger.getLogger(GitCommandProcessor.class);
    public static final String GIT_OUTPUT_ENCODING = "UTF-8";
    private static final String ENCODING_OPTION = "--encoding=UTF-8";
    private static final Pattern GIT_VERSION_PATTERN = Pattern.compile("^git version (.*)");
    private static final Pattern LS_REMOTE_LINE_PATTERN = Pattern.compile("^([0-9a-f]{40})\\s+(.*)");
    private static final String SSH_OPTIONS = "-o StrictHostKeyChecking=no -o BatchMode=yes -o UserKnownHostsFile=/dev/null";
    private static final String SSH_WIN = "@ssh -o StrictHostKeyChecking=no -o BatchMode=yes -o UserKnownHostsFile=/dev/null %*\r\n";
    private static final String SSH_UNIX = "#!/bin/sh\nexec ssh -o StrictHostKeyChecking=no -o BatchMode=yes -o UserKnownHostsFile=/dev/null $@\n";
    private static final String REMOTE_ORIGIN = "refs/remotes/origin/";
    private final String gitExecutable;
    private final BuildLogger buildLogger;
    private final String passwordToObfuscate;
    private final int commandTimeoutInMinutes;
    private final boolean maxVerboseOutput;
    private String proxyErrorMessage;
    private Throwable proxyException;
    private String sshCommand;
    private static final LoadingCache<GitExistencePair, String> GIT_EXISTENCE_CHECK_RESULT = CacheBuilder.newBuilder().expireAfterWrite(15L, TimeUnit.MINUTES).build((CacheLoader)new CacheLoader<GitExistencePair, String>(){

        public String load(GitExistencePair input) throws Exception {
            GitCommandProcessor gitCommandProcessor = input.gitCommandProcessor;
            File workingDirectory = input.workingDirectory;
            GitCommandBuilder commandBuilder = gitCommandProcessor.createCommandBuilder("version");
            GitStringOutputHandler outputHandler = new GitStringOutputHandler();
            int exitCode = gitCommandProcessor.runCommand(commandBuilder, workingDirectory, outputHandler);
            String output = outputHandler.getOutput();
            Matcher matcher = GIT_VERSION_PATTERN.matcher(output);
            if (!matcher.find()) {
                throw new GitCommandException("Unable to parse git command output: " + exitCode, null, output, "", null);
            }
            return matcher.group();
        }
    });

    public GitCommandProcessor(@NotNull String gitExecutable, @NotNull BuildLogger buildLogger, @Nullable String passwordToObfuscate, int commandTimeoutInMinutes, boolean maxVerboseOutput) {
        this.gitExecutable = gitExecutable;
        this.buildLogger = buildLogger;
        this.passwordToObfuscate = passwordToObfuscate;
        Preconditions.checkArgument((commandTimeoutInMinutes > 0 ? 1 : 0) != 0, (Object)"Command timeout must be greater than 0");
        this.commandTimeoutInMinutes = commandTimeoutInMinutes;
        this.maxVerboseOutput = maxVerboseOutput;
    }

    private String getDefaultSshWrapperScriptContent() {
        return SystemUtils.IS_OS_WINDOWS ? SSH_WIN : SSH_UNIX;
    }

    private String getCustomisedSshWrapperScriptContent() {
        return SystemUtils.IS_OS_WINDOWS ? "@\"" + this.sshCommand + "\" %*\r\n" : "#!/bin/sh\n\"" + this.sshCommand + "\" $@\n";
    }

    private String getSshScriptToRun() {
        String scriptContent = StringUtils.isBlank((String)this.sshCommand) ? this.getDefaultSshWrapperScriptContent() : this.getCustomisedSshWrapperScriptContent();
        try {
            boolean tmpDirHasSpaceInName = SystemUtils.getJavaIoTmpDir().getAbsolutePath().contains(" ");
            SharedTemporaryFiles.FileSpecBuilder specBuilder = SharedTemporaryFiles.builder((String)scriptContent).setPrefix("bamboo-ssh.").setSuffix(BambooFilenameUtils.getScriptSuffix()).setExecutable(true).setPrefer83PathsOnWindows(tmpDirHasSpaceInName);
            File sshScript = SharedTemporaryFiles.create((SharedTemporaryFiles.SharedTemporaryFileSpec)specBuilder.build());
            return sshScript.getAbsolutePath();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void checkGitExistenceInSystem(@NotNull File workingDirectory) throws RepositoryException {
        boolean gitDependsOnWorkingDirectory = this.gitExecutable.trim().startsWith(".");
        File directory = gitDependsOnWorkingDirectory ? workingDirectory : new File("/");
        GitExistencePair cacheKey = new GitExistencePair(directory, this.gitExecutable, this);
        try {
            GIT_EXISTENCE_CHECK_RESULT.get((Object)cacheKey);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            RepositoryException re = (RepositoryException)((Object)Narrow.to((Object)cause, RepositoryException.class));
            if (re != null) {
                throw re;
            }
            throw new RepositoryException(cause);
        }
    }

    public void runInitCommand(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("init");
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public List<String> runStatusCommand(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("status", "--porcelain", "--untracked-files=no");
        LineOutputHandlerImpl gitOutputHandler = new LineOutputHandlerImpl();
        this.runCommand(commandBuilder, workingDirectory, gitOutputHandler);
        if (log.isDebugEnabled()) {
            log.debug((Object)("git status output: " + gitOutputHandler.getStdout()));
        }
        return gitOutputHandler.getLines();
    }

    public void runFetchCommand(@NotNull File workingDirectory, @NotNull GitRepositoryAccessData accessData, String refSpec, boolean useShallow) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("fetch", accessData.getRepositoryUrl(), refSpec, "--update-head-ok");
        if (useShallow) {
            commandBuilder.shallowClone();
        }
        File shallowFile = new File(new File(workingDirectory, ".git"), "shallow");
        if (!useShallow && shallowFile.exists()) {
            log.info((Object)String.format("Detected that the directory needs to be converted to a full clone: %s, branch: %s, working dir: %s", accessData.getRepositoryUrl(), accessData.getVcsBranch().getName(), workingDirectory.getAbsolutePath()));
            commandBuilder.append("--depth=99999999");
        }
        if (accessData.isVerboseLogs()) {
            commandBuilder.verbose(true);
            commandBuilder.append("--progress");
        }
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
        File fetchHeadFile = new File(new File(workingDirectory, ".git"), "FETCH_HEAD");
        try {
            if (!fetchHeadFile.exists() || StringUtils.isBlank((String)FileUtils.readFileToString((File)fetchHeadFile))) {
                throw new RepositoryException("fatal: FETCH_HEAD is empty after fetch.");
            }
        }
        catch (IOException e) {
            throw new RepositoryException("fatal: Error reading FETCH_HEAD file");
        }
    }

    public void runCloneCommand(@NotNull File workingDirectory, @NotNull String repositoryUrl, boolean useShallowClone, boolean verboseLogs) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("clone", repositoryUrl);
        commandBuilder.destination(workingDirectory.getAbsolutePath());
        if (useShallowClone) {
            commandBuilder.shallowClone();
        }
        if (verboseLogs) {
            commandBuilder.verbose(true);
            commandBuilder.append("--progress");
        }
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public void runLocalCloneCommand(@NotNull File workingDirectory, File cacheDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("clone", "file://" + cacheDirectory.getAbsolutePath());
        commandBuilder.append("-n");
        commandBuilder.append("--reference");
        commandBuilder.append(cacheDirectory.getAbsolutePath());
        commandBuilder.destination(workingDirectory.getAbsolutePath());
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public void runCheckoutCommand(@NotNull File workingDirectory, String revision, String configuredBranchName) throws RepositoryException {
        String possibleBranch = this.getPossibleBranchNameForCheckout(workingDirectory, revision, configuredBranchName);
        String destination = revision;
        if (StringUtils.isNotBlank((String)possibleBranch)) {
            destination = possibleBranch;
        }
        this.runCheckoutCommandForBranchOrRevision(workingDirectory, destination);
    }

    public void runCheckoutCommandForBranchOrRevision(@NotNull File workingDirectory, String destination) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("checkout", "-f", destination);
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public void runSubmoduleUpdateCommand(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("submodule", "update", "--init", "--recursive");
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    @NotNull
    public String getRevisionHash(@NotNull File workingDirectory, @NotNull String revision) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-1", ENCODING_OPTION, "--format=%H");
        commandBuilder.append(revision);
        GitStringOutputHandler outputHandler = new GitStringOutputHandler(GIT_OUTPUT_ENCODING);
        this.runCommand(commandBuilder, workingDirectory, outputHandler);
        return outputHandler.getOutput().trim();
    }

    private boolean isMatchingLocalRef(String refString, String branchName) {
        return refString.startsWith("refs/heads/") && StringUtils.removeStart((String)refString, (String)"refs/heads/").equals(branchName);
    }

    private boolean isMatchingRemoteRef(String refString, String branchName) {
        return refString.startsWith(REMOTE_ORIGIN) && StringUtils.removeStart((String)refString, (String)REMOTE_ORIGIN).equals(branchName);
    }

    public String getPossibleBranchNameForCheckout(File workingDirectory, String revision, String configuredBranchName) throws RepositoryException {
        String branchName = StringUtils.isBlank((String)configuredBranchName) ? "master" : configuredBranchName;
        GitCommandBuilder commandBuilder = this.createCommandBuilder("show-ref", branchName);
        LineOutputHandlerImpl outputHandler = new LineOutputHandlerImpl();
        this.runCommand(commandBuilder, workingDirectory, outputHandler);
        List<String> lines = outputHandler.getLines();
        if (log.isDebugEnabled()) {
            log.debug((Object)"--- Full output: ---\n");
            for (String line : lines) {
                log.debug((Object)line);
            }
            log.debug((Object)"--- End of output: ---\n");
        }
        boolean remoteRefFound = false;
        for (String line : lines) {
            Iterable splitLine = Splitter.on((char)' ').trimResults().split((CharSequence)line.trim());
            String sha = (String)Iterables.getFirst((Iterable)splitLine, null);
            String refString = (String)Iterables.getLast((Iterable)splitLine, null);
            if (this.isMatchingLocalRef(refString, branchName)) {
                if (revision.equals(sha)) {
                    return branchName;
                }
                return "";
            }
            if (!this.isMatchingRemoteRef(refString, branchName) || !revision.equals(sha)) continue;
            remoteRefFound = true;
        }
        if (remoteRefFound) {
            return branchName;
        }
        return "";
    }

    @NotNull
    public ImmutableMap<String, String> getRemoteRefs(@NotNull File workingDirectory, @NotNull GitRepositoryAccessData accessData) throws RepositoryException {
        LineOutputHandlerImpl goh = new LineOutputHandlerImpl();
        GitCommandBuilder commandBuilder = this.createCommandBuilder("ls-remote", accessData.getRepositoryUrl());
        this.runCommand(commandBuilder, workingDirectory, goh);
        ImmutableMap<String, String> result = GitCommandProcessor.parseLsRemoteOutput(goh);
        return result;
    }

    @NotNull
    public String getBranchForSha(@NotNull File workingDirectory, String revision, String configuredBranch) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("branch", "-a", "--contains", revision);
        LineOutputHandlerImpl outputHandler = new LineOutputHandlerImpl();
        this.runCommand(commandBuilder, workingDirectory, outputHandler);
        String anyBranch = "";
        for (String branch : outputHandler.getLines()) {
            Iterable tokens = Splitter.on((String)" ").omitEmptyStrings().trimResults().split((CharSequence)branch);
            anyBranch = (String)Iterables.getLast((Iterable)tokens);
            if (!StringUtils.equals((String)anyBranch, (String)configuredBranch)) continue;
            return anyBranch;
        }
        return anyBranch;
    }

    @NotNull
    static ImmutableMap<String, String> parseLsRemoteOutput(LineOutputHandlerImpl goh) {
        ImmutableMap.Builder refs = ImmutableMap.builder();
        for (String ref : goh.getLines()) {
            Matcher matcher;
            log.debug((Object)("[" + ref + "]"));
            if (ref.contains("^{}") || !(matcher = LS_REMOTE_LINE_PATTERN.matcher(ref)).matches()) continue;
            refs.put((Object)matcher.group(2), (Object)matcher.group(1));
        }
        return refs.build();
    }

    public void setOrigin(@NotNull String originUrl, @NotNull File workingDirectory) throws RepositoryException {
        this.runCommand(this.createCommandBuilder("remote", "set-url", "origin", originUrl), workingDirectory);
    }

    public void addOrigin(@NotNull String originUrl, @NotNull File workingDirectory) throws RepositoryException {
        this.runCommand(this.createCommandBuilder("remote", "add", "origin", originUrl), workingDirectory);
    }

    public void setOriginToLocalDirectory(@NotNull File cacheDirectory, @NotNull File workingDirectory) throws RepositoryException {
        this.setOrigin("file://" + cacheDirectory.getAbsolutePath(), workingDirectory);
    }

    public GitCommandBuilder createCommandBuilder(String ... commands) {
        return new GitCommandBuilder(commands).executable(this.gitExecutable).sshCommand(this.getSshScriptToRun());
    }

    public void reportProxyError(String message, Throwable exception) {
        this.proxyErrorMessage = message;
        this.proxyException = exception;
    }

    public void runCommand(@NotNull GitCommandBuilder commandBuilder, @NotNull File workingDirectory) throws RepositoryException {
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public int runCommand(@NotNull GitCommandBuilder commandBuilder, @NotNull File workingDirectory, @NotNull GitOutputHandler outputHandler) throws RepositoryException {
        workingDirectory.mkdirs();
        BambooProcessHandler handler = new BambooProcessHandler((OutputHandler)outputHandler, (OutputHandler)outputHandler);
        List<String> commandArgs = commandBuilder.build();
        String maskedCommandLine = PasswordMaskingUtils.mask((String)BambooStringUtils.toCommandLineString(commandArgs), (String)this.passwordToObfuscate);
        if (this.maxVerboseOutput || log.isDebugEnabled()) {
            if (this.maxVerboseOutput) {
                this.buildLogger.addBuildLogEntry(maskedCommandLine);
            }
            log.debug((Object)("Running in " + workingDirectory + ": [" + maskedCommandLine + "]"));
        }
        ExternalProcessBuilder externalProcessBuilder = new ExternalProcessBuilder().command(commandArgs, workingDirectory).handler((ProcessHandler)handler).idleTimeout(TimeUnit.MINUTES.toMillis(this.commandTimeoutInMinutes)).env(commandBuilder.getEnv());
        ExternalProcess process = externalProcessBuilder.build();
        process.execute();
        if (!handler.succeeded()) {
            String maskedOutput = PasswordMaskingUtils.mask((String)outputHandler.getStdout(), (String)this.passwordToObfuscate);
            String message = "command " + maskedCommandLine + " failed with code " + handler.getExitCode() + ". Working directory was [" + workingDirectory + "].";
            Throwable cause = this.proxyException != null ? this.proxyException : handler.getException();
            throw new GitCommandException(message, cause, maskedOutput, this.proxyErrorMessage != null ? PasswordMaskingUtils.mask((String)this.proxyErrorMessage, (String)this.passwordToObfuscate) : maskedOutput, this.passwordToObfuscate);
        }
        return handler.getExitCode();
    }

    public void runMergeCommand(@NotNull GitCommandBuilder commandBuilder, @NotNull File workspaceDir) throws RepositoryException {
        LoggingOutputHandler mergeOutputHandler = new LoggingOutputHandler(this.buildLogger);
        this.runCommand(commandBuilder, workspaceDir, mergeOutputHandler);
        log.debug((Object)mergeOutputHandler.getStdout());
    }

    @NotNull
    public CommitContext extractCommit(File directory, String targetRevision) throws RepositoryException {
        CommitOutputHandler coh = new CommitOutputHandler(Collections.<String>emptySet());
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-1", ENCODING_OPTION, "--format=[d31bfa5_BAM_hash]%H%n[d31bfa5_BAM_author_name]%aN%n[d31bfa5_BAM_author_email]%ae%n[d31bfa5_BAM_timestamp]%ct%n[d31bfa5_BAM_commit_message]%s%n%b[d31bfa5_BAM_commit_message_end]%n[d31bfa5_BAM_file_list]", targetRevision);
        this.runCommand(commandBuilder, directory, coh);
        List<CommitContext> commits = coh.getExtractedCommits();
        if (commits.isEmpty()) {
            throw new RepositoryException("Could not find commit with revision " + targetRevision);
        }
        return commits.get(0);
    }

    public Pair<List<CommitContext>, Integer> runLogCommand(File cacheDirectory, String lastVcsRevisionKey, String targetRevision, @NotNull Set<String> shallows, int maxCommits) throws RepositoryException {
        String ancestryPathQuery = null;
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-p", "--name-only", ENCODING_OPTION, "--format=[d31bfa5_BAM_hash]%H%n[d31bfa5_BAM_author_name]%aN%n[d31bfa5_BAM_author_email]%ae%n[d31bfa5_BAM_timestamp]%ct%n[d31bfa5_BAM_commit_message]%s%n%b[d31bfa5_BAM_commit_message_end]%n[d31bfa5_BAM_file_list]");
        if (lastVcsRevisionKey.equals(targetRevision)) {
            commandBuilder.append(targetRevision).append("-1");
        } else {
            ancestryPathQuery = lastVcsRevisionKey + ".." + targetRevision;
            commandBuilder.append(ancestryPathQuery);
        }
        log.info((Object)("from revision: [" + lastVcsRevisionKey + "]; to revision: [" + targetRevision + "]"));
        CommitOutputHandler coh = new CommitOutputHandler(shallows, maxCommits);
        this.runCommand(commandBuilder, cacheDirectory, coh);
        if (ancestryPathQuery != null) {
            try {
                AncestryPathOutputHandler ancestryPathOutputHandler = new AncestryPathOutputHandler(maxCommits);
                GitCommandBuilder ancestryPathCommandBuilder = this.createCommandBuilder("log", "--format=%H", "--ancestry-path", ancestryPathQuery);
                this.runCommand(ancestryPathCommandBuilder, cacheDirectory, ancestryPathOutputHandler);
                for (Commit commit : Narrow.iterableDownTo(coh.getExtractedCommits(), Commit.class)) {
                    commit.setForeignCommit(!ancestryPathOutputHandler.getCommitsOnAncestryPath().contains(commit.getChangeSetId()));
                }
            }
            catch (RepositoryException re) {
                log.debug((Object)"git log --ancestry-path failed", (Throwable)re);
            }
        }
        return Pair.make(coh.getExtractedCommits(), (Object)coh.getSkippedCommitCount());
    }

    public void setSshCommand(String sshCommand) {
        this.sshCommand = sshCommand;
    }

    static class LoggingOutputHandler
    extends LineOutputHandler
    implements GitOutputHandler {
        final BuildLogger buildLogger;
        final StringBuffer stringBuffer;

        public LoggingOutputHandler(@NotNull BuildLogger buildLogger) {
            this.buildLogger = buildLogger;
            this.stringBuffer = new StringBuffer();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void processLine(int i, String s) {
            this.buildLogger.addBuildLogEntry(s);
            StringBuffer stringBuffer = this.stringBuffer;
            synchronized (stringBuffer) {
                if (this.stringBuffer.length() != 0) {
                    this.stringBuffer.append("\n");
                }
                this.stringBuffer.append(s);
            }
        }

        @Override
        public String getStdout() {
            return this.stringBuffer.toString();
        }
    }

    static class LineOutputHandlerImpl
    extends LineOutputHandler
    implements GitOutputHandler {
        private final List<String> lines = Collections.synchronizedList(Lists.newLinkedList());
        private final Joiner stdoutJoiner;

        public LineOutputHandlerImpl() {
            this(Joiner.on((String)"\n"));
        }

        public LineOutputHandlerImpl(@NotNull Joiner stdoutJoiner) {
            this.stdoutJoiner = stdoutJoiner;
        }

        protected void processLine(int i, String s) {
            this.lines.add(s);
        }

        @NotNull
        public List<String> getLines() {
            return this.lines;
        }

        @Override
        public String getStdout() {
            return this.stdoutJoiner.join(Iterables.filter(this.lines, (Predicate)BambooPredicates.stringIsNotEmpty()));
        }
    }

    static class GitStringOutputHandler
    extends StringOutputHandler
    implements GitOutputHandler {
        public GitStringOutputHandler(String encoding) {
            super(encoding);
        }

        @Deprecated
        public GitStringOutputHandler() {
        }

        @Override
        public String getStdout() {
            return this.getOutput();
        }
    }

    static interface GitOutputHandler
    extends OutputHandler {
        public String getStdout();
    }

    private static final class GitExistencePair {
        public final File workingDirectory;
        private final String gitExecutableName;
        public final GitCommandProcessor gitCommandProcessor;

        public GitExistencePair(File workingDirectory, String gitExecutableName, GitCommandProcessor gitCommandProcessor) {
            this.workingDirectory = workingDirectory;
            this.gitExecutableName = gitExecutableName;
            this.gitCommandProcessor = gitCommandProcessor;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            GitExistencePair that = (GitExistencePair)o;
            if (!this.gitExecutableName.equals(that.gitExecutableName)) {
                return false;
            }
            return this.workingDirectory.equals(that.workingDirectory);
        }

        public int hashCode() {
            int result = this.workingDirectory.hashCode();
            result = 31 * result + this.gitExecutableName.hashCode();
            return result;
        }
    }
}

