/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bamboo.plugins.git;

import com.atlassian.bamboo.build.logger.BuildLogger;
import com.atlassian.bamboo.commit.CommitContext;
import com.atlassian.bamboo.core.RepositoryUrlObfuscator;
import com.atlassian.bamboo.plugins.git.CommitOutputHandler;
import com.atlassian.bamboo.plugins.git.GitCommandBuilder;
import com.atlassian.bamboo.plugins.git.GitCommandException;
import com.atlassian.bamboo.plugins.git.GitRepository;
import com.atlassian.bamboo.repository.RepositoryException;
import com.atlassian.bamboo.ssh.ProxyErrorReceiver;
import com.atlassian.bamboo.util.BambooFileUtils;
import com.atlassian.bamboo.util.BambooFilenameUtils;
import com.atlassian.bamboo.utils.Pair;
import com.atlassian.utils.process.ExternalProcess;
import com.atlassian.utils.process.ExternalProcessBuilder;
import com.atlassian.utils.process.LineOutputHandler;
import com.atlassian.utils.process.OutputHandler;
import com.atlassian.utils.process.PluggableProcessHandler;
import com.atlassian.utils.process.ProcessHandler;
import com.atlassian.utils.process.StringOutputHandler;
import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.SystemUtils;
import org.apache.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

class GitCommandProcessor
implements Serializable,
ProxyErrorReceiver {
    private static final Logger log = Logger.getLogger(GitRepository.class);
    private static final Pattern GIT_VERSION_PATTERN = Pattern.compile("^git version (.*)");
    private static final String SSH_OPTIONS = "-o StrictHostKeyChecking=no -o BatchMode=yes -o UserKnownHostsFile=/dev/null";
    private static final String SSH_WIN = "@ssh -o StrictHostKeyChecking=no -o BatchMode=yes -o UserKnownHostsFile=/dev/null %*\r\n";
    private static final String SSH_UNIX = "#!/bin/sh\nexec ssh -o StrictHostKeyChecking=no -o BatchMode=yes -o UserKnownHostsFile=/dev/null $@\n";
    private final String gitExecutable;
    private final BuildLogger buildLogger;
    private final int commandTimeoutInMinutes;
    private final boolean maxVerboseOutput;
    private String proxyErrorMessage;
    private Throwable proxyException;
    private String sshCommand;

    public GitCommandProcessor(@Nullable String gitExecutable, @NotNull BuildLogger buildLogger, int commandTimeoutInMinutes, boolean maxVerboseOutput) {
        this.gitExecutable = gitExecutable;
        this.buildLogger = buildLogger;
        Preconditions.checkArgument((commandTimeoutInMinutes > 0 ? 1 : 0) != 0, (Object)"Command timeout must be greater than 0");
        this.commandTimeoutInMinutes = commandTimeoutInMinutes;
        this.maxVerboseOutput = maxVerboseOutput;
    }

    private String getDefaultSshWrapperScriptContent() {
        return SystemUtils.IS_OS_WINDOWS ? SSH_WIN : SSH_UNIX;
    }

    private String getCustomisedSshWrapperScriptContent() {
        return SystemUtils.IS_OS_WINDOWS ? "@\"" + this.sshCommand + "\" %*\r\n" : "#!/bin/sh\n\"" + this.sshCommand + "\" $@\n";
    }

    private String getSshScriptToRun() {
        String scriptContent = StringUtils.isBlank((String)this.sshCommand) ? this.getDefaultSshWrapperScriptContent() : this.getCustomisedSshWrapperScriptContent();
        try {
            File sshScript = BambooFileUtils.getSharedTemporaryFile((String)scriptContent, (String)"bamboo-ssh.", (String)BambooFilenameUtils.getScriptSuffix(), (boolean)true, null);
            return sshScript.getAbsolutePath();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public void checkGitExistenceInSystem(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("version");
        GitStringOutputHandler outputHandler = new GitStringOutputHandler();
        try {
            int exitCode = this.runCommand(commandBuilder, workingDirectory, outputHandler);
            String output = outputHandler.getOutput();
            Matcher matcher = GIT_VERSION_PATTERN.matcher(output);
            if (!matcher.find()) {
                String errorMessage = "Git Executable capability `" + this.gitExecutable + "' does not seem to be a git client. Is it properly set?";
                log.error((Object)(errorMessage + " Exit code: " + exitCode + " Output:\n[" + output + "]"));
                throw new RepositoryException(errorMessage);
            }
        }
        catch (GitCommandException e) {
            throw new RepositoryException("Git not found. Is Git Executable capability properly set in configuration?", (Throwable)((Object)e));
        }
    }

    public void runInitCommand(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("init");
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public List<String> runStatusCommand(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("status", "--porcelain", "--untracked-files=no");
        LineOutputHandlerImpl gitOutputHandler = new LineOutputHandlerImpl();
        this.runCommand(commandBuilder, workingDirectory, gitOutputHandler);
        log.debug((Object)("git status output: " + gitOutputHandler.getStdout()));
        return gitOutputHandler.getLines();
    }

    public void runFetchCommand(@NotNull File workingDirectory, @NotNull GitRepository.GitRepositoryAccessData accessData, String refSpec, boolean useShallow) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("fetch", accessData.repositoryUrl, refSpec, "--update-head-ok");
        if (useShallow) {
            commandBuilder.shallowClone();
        }
        File shallowFile = new File(new File(workingDirectory, ".git"), "shallow");
        if (!useShallow && shallowFile.exists()) {
            commandBuilder.append("--depth=99999999");
        }
        if (accessData.verboseLogs) {
            commandBuilder.verbose(true);
            commandBuilder.append("--progress");
        }
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public void runCloneCommand(@NotNull File workingDirectory, @NotNull String repositoryUrl, boolean useShallowClone, boolean verboseLogs) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("clone", repositoryUrl);
        commandBuilder.destination(workingDirectory.getAbsolutePath());
        if (useShallowClone) {
            commandBuilder.shallowClone();
        }
        if (verboseLogs) {
            commandBuilder.verbose(true);
            commandBuilder.append("--progress");
        }
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public void runCheckoutCommand(@NotNull File workingDirectory, String revision) throws RepositoryException {
        String possibleBranch = this.getPossibleBranchNameForCheckout(workingDirectory, revision);
        String destination = revision;
        if (StringUtils.isNotBlank((String)possibleBranch)) {
            destination = possibleBranch;
        }
        this.runCheckoutCommandForBranchOrRevision(workingDirectory, destination);
    }

    public void runCheckoutCommandForBranchOrRevision(@NotNull File workingDirectory, String destination) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("checkout", "-f", destination);
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public void runSubmoduleUpdateCommand(@NotNull File workingDirectory) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("submodule", "update", "--init", "--recursive");
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    @NotNull
    public String getRevisionHash(@NotNull File workingDirectory, @NotNull String revision) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-1", "--format=%H");
        commandBuilder.append(revision);
        GitStringOutputHandler outputHandler = new GitStringOutputHandler();
        this.runCommand(commandBuilder, workingDirectory, outputHandler);
        return outputHandler.getOutput().trim();
    }

    public String getPossibleBranchNameForCheckout(File workingDirectory, String revision) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-1", "--format=%d", "--decorate=full");
        commandBuilder.append(revision);
        GitStringOutputHandler outputHandler = new GitStringOutputHandler();
        this.runCommand(commandBuilder, workingDirectory, outputHandler);
        String revisionDescription = outputHandler.getOutput();
        if (StringUtils.isNotBlank((String)revisionDescription)) {
            HashSet possibleBranches = Sets.newHashSet((Iterable)Splitter.on((char)',').trimResults().split((CharSequence)CharMatcher.anyOf((CharSequence)"()").removeFrom((CharSequence)StringUtils.trim((String)revisionDescription))));
            for (String possibleBranch : possibleBranches) {
                if (!possibleBranch.startsWith("refs/heads/")) continue;
                return StringUtils.removeStart((String)possibleBranch, (String)"refs/heads/");
            }
        }
        return "";
    }

    @Nullable
    public String getRemoteBranchLatestCommitHash(File workingDirectory, GitRepository.GitRepositoryAccessData accessData, String branchRef) throws RepositoryException {
        LineOutputHandlerImpl goh = new LineOutputHandlerImpl();
        GitCommandBuilder commandBuilder = this.createCommandBuilder("ls-remote", accessData.repositoryUrl, branchRef);
        this.runCommand(commandBuilder, workingDirectory, goh);
        for (String ref : goh.getLines()) {
            if (!ref.contains(branchRef)) continue;
            return ref.substring(0, ref.indexOf(branchRef)).trim();
        }
        return null;
    }

    public Set<String> getRemoteRefs(File workingDirectory, GitRepository.GitRepositoryAccessData accessData) throws RepositoryException {
        LineOutputHandlerImpl goh = new LineOutputHandlerImpl();
        GitCommandBuilder commandBuilder = this.createCommandBuilder("ls-remote", accessData.repositoryUrl);
        this.runCommand(commandBuilder, workingDirectory, goh);
        HashSet result = Sets.newHashSet();
        for (String ref : goh.getLines()) {
            if (ref.contains("^{}") || !ref.contains("refs")) continue;
            result.add(ref.substring(ref.indexOf("refs")));
        }
        return result;
    }

    public GitCommandBuilder createCommandBuilder(String ... commands) {
        return new GitCommandBuilder(commands).executable(this.gitExecutable).sshCommand(this.getSshScriptToRun());
    }

    public void reportProxyError(String message, Throwable exception) {
        this.proxyErrorMessage = message;
        this.proxyException = exception;
    }

    public void runCommand(@NotNull GitCommandBuilder commandBuilder, @NotNull File workingDirectory) throws RepositoryException {
        this.runCommand(commandBuilder, workingDirectory, new LoggingOutputHandler(this.buildLogger));
    }

    public int runCommand(@NotNull GitCommandBuilder commandBuilder, @NotNull File workingDirectory, @NotNull GitOutputHandler outputHandler) throws RepositoryException {
        workingDirectory.mkdirs();
        PluggableProcessHandler handler = new PluggableProcessHandler();
        handler.setOutputHandler((OutputHandler)outputHandler);
        handler.setErrorHandler((OutputHandler)outputHandler);
        List<String> commandArgs = commandBuilder.build();
        if (this.maxVerboseOutput || log.isDebugEnabled()) {
            StringBuilder stringBuilder = new StringBuilder();
            for (String s : RepositoryUrlObfuscator.obfuscatePasswordsInUrls(commandArgs)) {
                stringBuilder.append(s).append(" ");
            }
            if (this.maxVerboseOutput) {
                this.buildLogger.addBuildLogEntry(stringBuilder.toString());
            }
            log.debug((Object)stringBuilder.toString());
        }
        ExternalProcessBuilder externalProcessBuilder = new ExternalProcessBuilder().command(commandArgs, workingDirectory).handler((ProcessHandler)handler).env(commandBuilder.getEnv());
        ExternalProcess process = externalProcessBuilder.build();
        process.setTimeout(TimeUnit.MINUTES.toMillis(this.commandTimeoutInMinutes));
        process.execute();
        if (!handler.succeeded()) {
            String stdout = RepositoryUrlObfuscator.obfuscatePasswordInUrl((String)outputHandler.getStdout());
            throw new GitCommandException("command " + RepositoryUrlObfuscator.obfuscatePasswordsInUrls(commandArgs) + " failed with code " + handler.getExitCode() + "." + " Working directory was [" + workingDirectory + "].", this.proxyException != null ? this.proxyException : handler.getException(), stdout, this.proxyErrorMessage != null ? "SSH Proxy error: " + this.proxyErrorMessage : stdout);
        }
        return handler.getExitCode();
    }

    public void runMergeCommand(@NotNull GitCommandBuilder commandBuilder, @NotNull File workspaceDir) throws RepositoryException {
        LoggingOutputHandler mergeOutputHandler = new LoggingOutputHandler(this.buildLogger);
        this.runCommand(commandBuilder, workspaceDir, mergeOutputHandler);
        log.debug((Object)mergeOutputHandler.getStdout());
    }

    public CommitContext extractCommit(File directory, String targetRevision) throws RepositoryException {
        CommitOutputHandler coh = new CommitOutputHandler(Collections.<String>emptySet());
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-1", "--format=[d31bfa5_BAM_hash]%H%n[d31bfa5_BAM_commiter_name]%cN%n[d31bfa5_BAM_commiter_email]%ce%n[d31bfa5_BAM_timestamp]%ct%n[d31bfa5_BAM_commit_message]%s%n%b[d31bfa5_BAM_file_list]", targetRevision);
        this.runCommand(commandBuilder, directory, coh);
        List<CommitContext> commits = coh.getExtractedCommits();
        if (commits.isEmpty()) {
            throw new RepositoryException("Could not find commit with revision " + targetRevision);
        }
        return commits.get(0);
    }

    public Pair<List<CommitContext>, Integer> runLogCommand(File cacheDirectory, String lastVcsRevisionKey, String targetRevision, @NotNull Set<String> shallows, int maxCommits) throws RepositoryException {
        GitCommandBuilder commandBuilder = this.createCommandBuilder("log", "-p", "--name-only", "--format=[d31bfa5_BAM_hash]%H%n[d31bfa5_BAM_commiter_name]%cN%n[d31bfa5_BAM_commiter_email]%ce%n[d31bfa5_BAM_timestamp]%ct%n[d31bfa5_BAM_commit_message]%s%n%b[d31bfa5_BAM_file_list]");
        if (lastVcsRevisionKey.equals(targetRevision)) {
            commandBuilder.append(targetRevision).append("-1");
        } else {
            commandBuilder.append(lastVcsRevisionKey + ".." + targetRevision);
        }
        log.info((Object)("from revision: [" + lastVcsRevisionKey + "]; to revision: [" + targetRevision + "]"));
        CommitOutputHandler coh = new CommitOutputHandler(shallows, maxCommits);
        this.runCommand(commandBuilder, cacheDirectory, coh);
        return new Pair(coh.getExtractedCommits(), (Object)coh.getSkippedCommitCount());
    }

    public void setSshCommand(String sshCommand) {
        this.sshCommand = sshCommand;
    }

    static class LoggingOutputHandler
    extends LineOutputHandler
    implements GitOutputHandler {
        final BuildLogger buildLogger;
        final StringBuilder stringBuilder;

        public LoggingOutputHandler(@NotNull BuildLogger buildLogger) {
            this.buildLogger = buildLogger;
            this.stringBuilder = new StringBuilder();
        }

        protected void processLine(int i, String s) {
            this.buildLogger.addBuildLogEntry(s);
            if (this.stringBuilder.length() != 0) {
                this.stringBuilder.append("\n");
            }
            this.stringBuilder.append(s);
        }

        @Override
        public String getStdout() {
            return this.stringBuilder.toString();
        }
    }

    static class LineOutputHandlerImpl
    extends LineOutputHandler
    implements GitOutputHandler {
        private final List<String> lines = Lists.newLinkedList();

        LineOutputHandlerImpl() {
        }

        protected void processLine(int i, String s) {
            this.lines.add(s);
        }

        @NotNull
        public List<String> getLines() {
            return this.lines;
        }

        @Override
        public String getStdout() {
            return this.lines.toString();
        }
    }

    static class GitStringOutputHandler
    extends StringOutputHandler
    implements GitOutputHandler {
        GitStringOutputHandler() {
        }

        @Override
        public String getStdout() {
            return this.getOutput();
        }
    }

    static interface GitOutputHandler
    extends OutputHandler {
        public String getStdout();
    }
}

