package com.atlassian.aws.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerConfiguration;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;

import static com.amazonaws.services.s3.internal.Constants.GB;
import static com.amazonaws.services.s3.internal.Constants.MB;

public class EtagCalculator
{
    private static final long DEFAULT_MULTIPART_UPLOAD_THRESHOLD = 16 * MB;
    private static final long DEFAULT_MINIMUM_UPLOAD_PART_SIZE = 5 * MB;

    private static final long DEFAULT_MULTIPART_COPY_THRESHOLD = 5 * GB;
    private static final long DEFAULT_MINIMUM_COPY_PART_SIZE = 100 * MB;

    private final long multipartUploadThreshold;
    private final long minimumUploadPartSize;

    public EtagCalculator(final long multipartUploadThreshold, final long minimumUploadPartSize)
    {
        this.multipartUploadThreshold = multipartUploadThreshold;
        this.minimumUploadPartSize = minimumUploadPartSize;
    }

    public String calculateEtag(@NotNull final File file) throws IOException
    {
        return S3UtilsThin.calculateEtag(file, multipartUploadThreshold, minimumUploadPartSize);
    }

    public static EtagCalculator forTransferManager(final TransferManager transferManager)
    {
        final TransferManagerConfiguration configuration = transferManager.getConfiguration();
        return new EtagCalculator(configuration.getMultipartUploadThreshold(), configuration.getMinimumUploadPartSize());
    }

    /**
     * Returns a transfer manager with known Etag calculation settings
     */
    public static TransferManager newCompatibleTransferManager(final AmazonS3 s3Client)
    {
        final TransferManager transferManager = new TransferManager(s3Client);
        final TransferManagerConfiguration configuration = transferManager.getConfiguration();
        configuration.setMultipartUploadThreshold(DEFAULT_MULTIPART_UPLOAD_THRESHOLD);
        configuration.setMinimumUploadPartSize(DEFAULT_MINIMUM_UPLOAD_PART_SIZE);

        configuration.setMultipartCopyThreshold(DEFAULT_MULTIPART_COPY_THRESHOLD);
        configuration.setMultipartCopyPartSize(DEFAULT_MINIMUM_COPY_PART_SIZE);
        return transferManager;
    }
}
