package com.atlassian.aws.utils;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.util.Base64;


public class JsonUtils
{
    static final String KEY_STORE_PASSWORD = "bamboo";
    
    private static final ObjectMapper mapper;

    static
    {
        final SimpleModule keyStoreModule = new SimpleModule()
                .addSerializer(KeyStore.class, new KeyStoreSerialiser())
                .addDeserializer(KeyStore.class, new KeyStoreDeserialiser());
        
        mapper = new ObjectMapper()
                .setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY)
                .registerModule(keyStoreModule);
    }

    public static <T> T fromJson(final String json, final Class<T> classOfT) throws IOException
    {
        return mapper.readValue(json, classOfT);
    }

    public static String toJson(final Object userData)
    {
        try
        {
            return mapper.writeValueAsString(userData);
        }
        catch (final JsonProcessingException e)
        {
            throw new RuntimeException(e);
        }
    }

    private static class KeyStoreSerialiser extends JsonSerializer<KeyStore>
    {
        @Override
        public void serialize(final KeyStore keyStore, final JsonGenerator jsonGenerator, final SerializerProvider serializerProvider) throws IOException
        {
            final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
            try
            {
                keyStore.store(byteArrayOutputStream, KEY_STORE_PASSWORD.toCharArray());
            }
            catch (final Exception e)
            {
                throw new IllegalStateException(e);
            }

            final String keyStoreContents = Base64.getMimeEncoder().encodeToString(byteArrayOutputStream.toByteArray());

            jsonGenerator.writeStartArray();
            jsonGenerator.writeString(keyStore.getType());
            jsonGenerator.writeString(keyStoreContents);
            jsonGenerator.writeEndArray();
        }
    }
    
    private static class KeyStoreDeserialiser extends JsonDeserializer<KeyStore>
    {
        @Override
        public KeyStore deserialize(final JsonParser jsonParser, final DeserializationContext deserializationContext) throws IOException
        {
            if (jsonParser.getCurrentToken() != JsonToken.START_ARRAY) 
            {
                throw new JsonParseException("Start array expected", jsonParser.getCurrentLocation());
            }

            try
            {
                final String keyStoreType = jsonParser.nextTextValue();
                final String keyStoreContent = jsonParser.nextTextValue();

                final KeyStore keyStore = KeyStore.getInstance(keyStoreType);

                final byte[] decoded = Base64.getMimeDecoder().decode(keyStoreContent);

                keyStore.load(new ByteArrayInputStream(decoded), KEY_STORE_PASSWORD.toCharArray());

                return keyStore;
            }
            catch (final Exception e)
            {
                throw new JsonParseException("Could not deserialize keystore", jsonParser.getCurrentLocation(), e);
            }
        }
    }
}
