package com.atlassian.aws.s3;

public final class S3Path
{
    private static final String S3_PROTOCOL = "s3://";
    private static final int PATH_COMPONENTS = 4;
    private final String bucket;
    private final String key;

    public S3Path(final String bucket, final String key)
    {
        this.bucket = bucket;
        this.key = stripTrailingSlash(key);
    }

    public S3Path(final String srcPath)
    {
        if (!isS3Path(srcPath))
        {
            throw new IllegalArgumentException(srcPath + " is not an S3 path");
        }
        final String[] split = srcPath.split("/", PATH_COMPONENTS);
        this.bucket = split[2];
        if (split.length == PATH_COMPONENTS)
        {
            this.key = stripTrailingSlash(split[3]);
        }
        else
        {
            this.key = "";
        }
    }

    private static String stripTrailingSlash(final String s)
    {
        return s.endsWith("/") ? s.substring(0, s.length()-1) : s;
    }

    private boolean isS3Path(final String srcPath)
    {
        return srcPath.startsWith(S3_PROTOCOL);
    }

    /**
     *
     * @deprecated since 5.10 use {@link #getKey()} instead
     */
    @Deprecated
    public String getDirectory()
    {
        return key;
    }

    public String getKey()
    {
        return key;
    }

    public String getBucket()
    {
        return bucket;
    }

    @Override
    public String toString()
    {
        return S3_PROTOCOL + bucket + '/' + key;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        S3Path s3Path = (S3Path) o;

        if (!bucket.equals(s3Path.bucket)) return false;
        return key.equals(s3Path.key);
    }

    @Override
    public int hashCode()
    {
        int result = bucket.hashCode();
        result = 31 * result + key.hashCode();
        return result;
    }
}
