package com.atlassian.audit.search;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.audit.ao.dao.entity.AoAuditEntity;
import com.atlassian.audit.ao.dao.entity.AoAuditEntityCategory;
import com.atlassian.audit.coverage.SingleValueCache;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import net.java.ao.Query;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 *  Query Categories from Audit DB and store it in a Time To Live cache.
 */
public class CategoriesProvider {
    private final ActiveObjects ao;
    private final TransactionTemplate transactionTemplate;
    private final SingleValueCache<Set<String>> categoriesCache;

    public CategoriesProvider(ActiveObjects ao,
                              TransactionTemplate transactionTemplate,
                              int refreshIntervalInSeconds) {
        this.ao = ao;
        this.transactionTemplate = transactionTemplate;
        this.categoriesCache = new SingleValueCache<>(
                this::queryDistinctCategories,
                refreshIntervalInSeconds, SECONDS);
    }

    /**
     * This method is invoked by {@link SingleValueCache} when cache expired to query all distinct category values.
     * Use full scope search is faster than timestamp scoped query, though the latter has less number of rows to scan.
     *
     * @return all distinct categories
     */
    @Nonnull
    private Set<String> queryDistinctCategories() {
        return Arrays.stream(transactionTemplate.execute(() -> ao.find(AoAuditEntityCategory.class,
                Query.select(AoAuditEntity.CATEGORY_COLUMN)
                        .distinct())))
                .map(AoAuditEntity::getCategory)
                .collect(Collectors.toSet());
    }

    public List<String> getCategories() {
        return categoriesCache.get().stream().sorted().collect(Collectors.toList());
    }
}
