package com.atlassian.audit.schedule.db.limit;

import com.atlassian.audit.ao.dao.AuditEntityDao;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;

import static com.atlassian.audit.ao.dao.AoAuditEntityDao.MIN_RETAIN_LIMIT_DEFAULT;
import static com.atlassian.audit.schedule.db.limit.DbLimiterScheduler.AUDIT_DB_LIMIT_ROWS_KEY;

/**
 * The job will periodically delete old records from audit table
 * if the total number exceeds a predefined limit (defaul to 10M)
 */
public class DbLimiterJobRunner implements JobRunner {

    private static final Logger log = LoggerFactory.getLogger(DbLimiterJobRunner.class);

    private final AuditEntityDao auditEntityDao;

    private final int rowsLimit;
    private final int rowsLimitBuffer;

    public DbLimiterJobRunner(AuditEntityDao auditEntityDao, int dbRowLimit, int rowsLimitBuffer) {
        this.auditEntityDao = auditEntityDao;
        this.rowsLimit = dbRowLimit;
        this.rowsLimitBuffer = rowsLimitBuffer;
    }

    @Nonnull
    @Override
    public JobRunnerResponse runJob(JobRunnerRequest jobRunnerRequest) {
        try {
            log.info("DbLimiterJobRunner Started");
            int rowsToRetain = rowsLimit - rowsLimitBuffer;
            if (rowsToRetain < MIN_RETAIN_LIMIT_DEFAULT) {
                log.warn("System property {}={} is too small, {} is used to truncate the audit DB",
                        AUDIT_DB_LIMIT_ROWS_KEY, rowsLimit, MIN_RETAIN_LIMIT_DEFAULT);
            }
            auditEntityDao.retainRecent(Math.max(rowsToRetain, MIN_RETAIN_LIMIT_DEFAULT));
            log.info("DbLimiterJobRunner Finished");
            return JobRunnerResponse.success();
        } catch (RuntimeException e) {
            log.error("Failed to execute DbLimiterJob", e);
            return JobRunnerResponse.failed(e);
        }
    }

}
