package com.atlassian.audit.retention;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.ao.dao.AuditEntityDao;
import com.atlassian.audit.api.AuditRetentionConfigService;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.time.Instant;
import java.time.Period;
import java.time.ZonedDateTime;
import java.util.concurrent.TimeUnit;

/**
 * The job will periodically cleanup audit table based on retention policy.
 */
public class RetentionJobRunner implements JobRunner {
    public static final String CLEAN_UP_INTERVAL_IN_HOURS_KEY = "plugin.audit.retention.interval.hours";
    public static final int CLEAN_UP_INTERVAL_IN_HOURS_DEFAULT = 23;

    private static final Logger log = LoggerFactory.getLogger(RetentionJobRunner.class);

    private static final String CLEAN_UP_LAST_RUN_KEY = "com.atlassian.audit.plugin:cleanup.last.run.start";
    private final AuditEntityDao auditEntityDao;
    private final AuditRetentionConfigService auditRetentionConfigService;
    private final PluginSettingsFactory settingsFactory;
    private final int cleanUpInterval;

    @VisibleForTesting
    private RetentionJobRunner(AuditEntityDao auditEntityDao, AuditRetentionConfigService auditRetentionConfigService,
                               PluginSettingsFactory settingsFactory) {
        this(auditEntityDao, auditRetentionConfigService, settingsFactory, CLEAN_UP_INTERVAL_IN_HOURS_DEFAULT);
    }

    public RetentionJobRunner(AuditEntityDao auditEntityDao, AuditRetentionConfigService auditRetentionConfigService,
                              PluginSettingsFactory settingsFactory, int cleanUpInterval) {
        this.auditEntityDao = auditEntityDao;
        this.auditRetentionConfigService = auditRetentionConfigService;
        this.settingsFactory = settingsFactory;
        this.cleanUpInterval = cleanUpInterval;
    }

    @Nullable
    @Override
    public JobRunnerResponse runJob(JobRunnerRequest jobRunnerRequest) {
        log.info("RetentionJobRunner Started");
        PluginSettings settings = settingsFactory.createGlobalSettings();
        long start = jobRunnerRequest.getStartTime().getTime();
        if (shouldRun((String) settings.get(CLEAN_UP_LAST_RUN_KEY), start)) {
            try {
                Period period = auditRetentionConfigService.getConfig().getPeriod();
                Instant before = ZonedDateTime.now()
                        .minusDays(period.getDays())
                        .minusMonths(period.getMonths())
                        .minusYears(period.getYears())
                        .toInstant();
                auditEntityDao.removeBefore(before);
                log.info("RetentionJobRunner Finished");
                return JobRunnerResponse.success();
            } catch (Exception e) {
                log.error("Failed to execute RetentionJob ", e);
                return JobRunnerResponse.failed(e);
            } finally {
                settings.put(CLEAN_UP_LAST_RUN_KEY, Long.toString(start));
            }
        } else {
            return JobRunnerResponse.success();
        }
    }

    private boolean shouldRun(String lastRun, long start) {
        if (StringUtils.isNumeric(lastRun)) {
            //If lastRun is numeric, parse it and use it to determine whether the last run was "too soon"
            //for cleanup to run again
            boolean shouldRun = Long.parseLong(lastRun) + TimeUnit.HOURS.toMillis(cleanUpInterval) < start;
            if (!shouldRun) {
                log.trace("Last clean up job was within {} hours, skipping this run.", cleanUpInterval);
            }
            return shouldRun;
        }
        return true;
    }
}
