package com.atlassian.audit.rest.validation;

import com.atlassian.audit.rest.validation.exception.InvalidQueryException;
import com.atlassian.plugins.rest.common.interceptor.MethodInvocation;
import com.atlassian.plugins.rest.common.interceptor.ResourceInterceptor;

import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MultivaluedMap;
import java.lang.reflect.InvocationTargetException;

/**
 * Invokes the associated {@link Validator} for each {@link QueryParam} and throws {@link InvalidQueryException}
 * if any issues are found
 */
public class ValidationInterceptor implements ResourceInterceptor {
    @Override
    public void intercept(MethodInvocation methodInvocation) throws InvocationTargetException, IllegalAccessException {
        MultivaluedMap<String, String> queryParameters = methodInvocation.getHttpContext().getUriInfo().getQueryParameters();
        ValidationResult validationResult = new ValidationResult();

        methodInvocation.getMethod().getParameters().stream()
                .filter(p -> p.isAnnotationPresent(QueryParam.class))
                .filter(p -> p.isAnnotationPresent(Validator.class))
                .forEach(parameter -> {
                            final Class<? extends QueryParamValidator> validatorClass =
                                    parameter.getAnnotation(Validator.class).value();
                            final QueryParamValidator parameterValidator;
                            try {
                                parameterValidator = validatorClass.getDeclaredConstructor().newInstance();
                            } catch (InstantiationException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
                                throw new RuntimeException("The class " + validatorClass + " should have default constructor");
                            }
                            String key = parameter.getAnnotation(QueryParam.class).value();
                            final String parameterValue = queryParameters.getFirst(key);
                            if (parameterValue != null) {
                                parameterValidator.validate(parameterValue, validationResult);
                            }
                        }
                );

        if (validationResult.hasErrors()) {
            throw new InvalidQueryException(validationResult);
        }
        methodInvocation.invoke();
    }
}
