package com.atlassian.audit.plugin.configuration;

import com.atlassian.sal.api.ApplicationProperties;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Finds and returns property values either from the system or from SAL to account for different product practises.
 * <p>
 * System properties take precedence over those provided by SAL.
 */
public class SystemThenSalPropertiesProvider implements PropertiesProvider {
    private final ApplicationProperties applicationProperties;

    public SystemThenSalPropertiesProvider(ApplicationProperties applicationProperties) {
        this.applicationProperties = applicationProperties;
    }

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns null if the property is not found.
     *
     * @param key the property key.
     * @return the value of the property or null.
     */
    @Override
    @Nullable
    public String getProperty(@Nonnull String key) {
        requireNonNull(key, "key to get property");
        final String systemPropertyValue = System.getProperty(key);
        if (systemPropertyValue != null) {
            return systemPropertyValue;
        }

        try {
            // Replace deprecated method when the new SAL API is created and supported by all products
            return applicationProperties.getPropertyValue(key);
        } catch (UnsupportedOperationException e) { // Confluence does not support #getPropertyValue
            return null;
        }
    }

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL.
     *
     * @param key the property key.
     * @return true if and only if the property is set to "true" (case insensitive)
     */
    @Override
    public boolean getBoolean(@Nonnull String key) {
        requireNonNull(key, "key to get boolean property");
        return Boolean.parseBoolean(this.getProperty(key));
    }

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns the default value argument if the property is not found.
     *
     * @param key          the property key.
     * @param defaultValue a default value.
     * @return the value of the property or the default value.
     */
    @Override
    @Nullable
    public String getProperty(@Nonnull String key, String defaultValue) {
        requireNonNull(key, "key to get property");
        final String propertyValue = this.getProperty(key);
        return propertyValue == null ? defaultValue : propertyValue;
    }

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns the default value argument if the property is not found.
     *
     * @param key          the property key.
     * @param defaultValue a default value.
     * @return the value of the property as an int or the default value.
     */
    @Override
    public int getInteger(@Nonnull String key, int defaultValue) {
        requireNonNull(key, "key to get integer property");
        try {
            return Integer.parseInt(getProperty(key));
        } catch (NumberFormatException | NullPointerException e) {
            return defaultValue;
        }
    }

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns the default value argument if the property is not found.
     *
     * @param key          the property key.
     * @param defaultValue a default value.
     * @return the value of the property as a long or the default value.
     */
    @Override
    public long getLong(@Nonnull String key, long defaultValue) {
        requireNonNull(key, "key to get long int property");
        try {
            return Long.parseLong(getProperty(key));
        } catch (NumberFormatException | NullPointerException e) {
            return defaultValue;
        }
    }
}
