package com.atlassian.audit.plugin.configuration;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Finds and returns property values.
 */
public interface PropertiesProvider {
    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns null if the property is not found.
     *
     * @param key the property key.
     * @return the value of the property or null.
     */
    @Nullable
    String getProperty(@Nonnull String key);

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL.
     *
     * @param key the property key.
     * @return true if and only if the property is set to "true" (case insensitive)
     */
    boolean getBoolean(@Nonnull String key);

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns the default value argument if the property is not found.
     *
     * @param key          the property key.
     * @param defaultValue a default value.
     * @return the value of the property or the default value.
     */
    @Nullable
    String getProperty(@Nonnull String key, String defaultValue);

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns the default value argument if the property is not found.
     *
     * @param key          the property key.
     * @param defaultValue a default value.
     * @return the value of the property as an int or the default value.
     */
    int getInteger(@Nonnull String key, int defaultValue);

    /**
     * Searches for the property with the specified key in the system properties and iff not found then using SAL. The
     * method returns the default value argument if the property is not found.
     *
     * @param key          the property key.
     * @param defaultValue a default value.
     * @return the value of the property as a long or the default value.
     */
    long getLong(@Nonnull String key, long defaultValue);
}
