package com.atlassian.audit.plugin.configuration;

import com.atlassian.audit.ao.dao.AuditEntityDao;
import com.atlassian.audit.ao.service.AnalyticsTrackedAuditSearchService;
import com.atlassian.audit.ao.service.AuditedSearchService;
import com.atlassian.audit.ao.service.CachedCountService;
import com.atlassian.audit.ao.service.DatabaseAuditSearchService;
import com.atlassian.audit.ao.service.RateLimitedSearchService;
import com.atlassian.audit.ao.service.RestrictiveSearchService;
import com.atlassian.audit.api.AuditSearchService;
import com.atlassian.audit.api.AuditService;
import com.atlassian.audit.permission.PermissionChecker;
import com.atlassian.audit.plugin.AuditPluginInfo;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.web.context.HttpContext;
import org.osgi.framework.ServiceRegistration;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import static com.atlassian.plugins.osgi.javaconfig.ExportOptions.as;
import static com.atlassian.plugins.osgi.javaconfig.OsgiServices.exportOsgiService;
import static com.atlassian.plugins.osgi.javaconfig.OsgiServices.importOsgiService;

@Configuration
public class AuditSearchConfiguration {

    @SuppressWarnings("rawtypes")
    @Bean
    public FactoryBean<ServiceRegistration> exportAuditSearchService(AuditSearchService auditSearchService) {
        return exportOsgiService(auditSearchService, as(AuditSearchService.class));
    }

    @Bean
    public HttpContext httpContextSupplier() {
        return importOsgiService(HttpContext.class);
    }

    @Bean
    public AuditSearchService auditSearchService(AuditEntityDao auditEntityDao,
                                                           PermissionChecker permissionChecker,
                                                           EventPublisher eventPublisher,
                                                           AuditPluginInfo auditPluginInfo,
                                                           AuditService auditService,
                                                           HttpContext httpContextSupplier,
                                                           PropertiesProvider propertiesProvider
    ) {
        final int maxConcurrentTextSearchRequests = propertiesProvider
                .getInteger("plugin.audit.search.max.concurrent.text.requests", 5);
        final int maxConcurrentNonTextSearchRequests = propertiesProvider
                .getInteger("plugin.audit.search.max.concurrent.nontext.requests", 10);
        final int queryTimeoutSeconds = propertiesProvider.getInteger("plugin.audit.search.query.timeout", 30);

        return new RestrictiveSearchService(permissionChecker,
                new AuditedSearchService(
                        new AnalyticsTrackedAuditSearchService(
                                new RateLimitedSearchService(maxConcurrentTextSearchRequests, maxConcurrentNonTextSearchRequests, queryTimeoutSeconds,
                                        new DatabaseAuditSearchService(auditEntityDao)), eventPublisher, auditPluginInfo, httpContextSupplier), auditService));
    }

    @Bean
    public CachedCountService countCachedSearchService(AuditEntityDao auditEntityDao,
                                                       PropertiesProvider propertiesProvider) {
        final int cacheExpirationInMinutes = propertiesProvider.getInteger("plugin.audit.database.usage.expiration.minutes", 60);
        return new CachedCountService(auditEntityDao, cacheExpirationInMinutes);
    }
}
