package com.atlassian.audit.lookup.rest;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditAuthor;
import com.atlassian.audit.entity.AuditResource;
import com.atlassian.audit.rest.model.AuditResourceLookupJson;
import com.atlassian.audit.rest.model.ResponseErrorJson;
import com.atlassian.audit.rest.utils.AuditEntitySerializer;
import com.atlassian.audit.spi.lookup.AuditingResourcesLookupService;
import com.atlassian.sal.api.ApplicationProperties;
import io.swagger.v3.oas.annotations.OpenAPIDefinition;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.info.Info;
import io.swagger.v3.oas.annotations.media.ArraySchema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;

import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.sal.api.UrlMode.CANONICAL;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;

@OpenAPIDefinition(info = @Info(title = "Audit Resource Lookup", version = "0.0.1", description = "API to lookup " +
        "data for Advanced Auditing UI, supports author, project, repository and space. The root path is " +
        "/rest/auditing/1.0"))
@Path("/lookup")
@Produces(APPLICATION_JSON)
public class AuditResourceLookupRestResource {

    private static final String DELIMS = ". ";

    private final AuditingResourcesLookupService auditResourceLookupProvider;
    private final ApplicationProperties applicationProperties;

    public AuditResourceLookupRestResource(AuditingResourcesLookupService auditResourceLookupProvider, ApplicationProperties applicationPropertiesSupplier) {
        this.auditResourceLookupProvider = auditResourceLookupProvider;
        this.applicationProperties = applicationPropertiesSupplier;
    }

    private static AuditResourceLookupJson toJson(AuditResource auditResource) {
        return new AuditResourceLookupJson(
                auditResource.getName(),
                auditResource.getType(),
                auditResource.getUri(),
                auditResource.getId()
        );
    }

    @GET
    @Path("/resource/{resourceType}")
    @Operation(summary = "Lookup audit resources", tags = {"resource", "lookup"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditResourceResponseJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response lookupResources(
            @Parameter(description = "The type of resource")
            @PathParam("resourceType")
                    String resType,
            @Parameter(description = "The number of records to skip")
            @Schema(minimum = "0")
            @QueryParam("offset")
            @DefaultValue("0")
                    int offset,
            @Parameter(description = "Location of last result returned in format of ID or resource. For making a request for page X, " +
                    "the value of this field can be obtained from pagingInfo->nextPageCursor in response for page X-1",
                    example = "9")
            @QueryParam("pageCursor")
                    String cursor,
            @Parameter(description = "The maximum number of records returned")
            @Schema(minimum = "1", maximum = "100000")
            @QueryParam("limit")
            @DefaultValue("100")
                    int limit,
            @Parameter(description = "Search text")
            @QueryParam("search")
                    String search,
            @Context UriInfo uriInfo) {
        final Page<AuditResource, String> res = auditResourceLookupProvider.lookupAuditResource(resType, search,
                new PageRequest.Builder<String>()
                        .offset(offset)
                        .limit(limit)
                        .cursor(cursor)
                        .build());
        final AuditResourceResponseJson response = new AuditResourceResponseJson(res, AuditResourceLookupRestResource::toJson, applicationProperties.getBaseUrl(CANONICAL), uriInfo);
        return Response.ok(response).build();
    }

    @GET
    @Path("/author")
    @Operation(summary = "Lookup audit authors", tags = {"author", "lookup"})
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Successful operation", content = @Content(schema =
            @Schema(implementation = AuditAuthorResponseJson.class))),
            @ApiResponse(responseCode = "400", description = "Bad request", content = @Content(array = @ArraySchema
                    (schema = @Schema(implementation = ResponseErrorJson.class))))})
    public Response lookupAuthors(
            @Parameter(description = "The number of records to skip")
            @Schema(minimum = "0")
            @QueryParam("offset")
            @DefaultValue("0")
                    int offset,
            @Parameter(description = "Location of last result returned in format of ID of author. For making a request for page X, " +
                    "the value of this field can be obtained from pagingInfo->nextPageCursor in response for page X-1",
                    example = "9")
            @QueryParam("pageCursor")
                    String cursor,
            @Parameter(description = "The maximum number of records returned")
            @Schema(minimum = "1", maximum = "100000")
            @QueryParam("limit")
            @DefaultValue("100")
                    int limit,
            @Parameter(description = "Search text")
            @QueryParam("search")
                    String search,
            @Context UriInfo uriInfo) {
        // Only inspect common authors for 1st page
        List<AuditAuthor> commonAuthors = (cursor != null || offset > 0) ? Collections.emptyList() : Stream.of(
                AuditAuthor.SYSTEM_AUTHOR, AuditAuthor.ANONYMOUS_AUTHOR, AuditAuthor.UNKNOWN_AUTHOR).filter(
                auditAuthor -> authorMatches(search, auditAuthor)).collect(Collectors.toList());

        final Page<AuditAuthor, String> res = auditResourceLookupProvider.lookupAuditAuthor(search,
                new PageRequest.Builder<String>()
                        .offset(offset)
                        .limit(Math.max(0, limit - commonAuthors.size()))
                        .cursor(cursor)
                        .build());

        final AuditAuthorResponseJson response = new AuditAuthorResponseJson(addCommonAuthors(res, commonAuthors), AuditEntitySerializer::toJson, applicationProperties.getBaseUrl(CANONICAL), uriInfo);
        return Response.ok(response).build();
    }

    private Page<AuditAuthor, String> addCommonAuthors(Page<AuditAuthor, String> page, List<AuditAuthor> commonAuthors) {
        List<AuditAuthor> auditAuthors = Stream.concat(
                commonAuthors.stream(),
                page.getValues().stream().filter(auditAuthor -> auditAuthor.getName() != null)
                        .sorted(Comparator.comparing(auditAuthor -> auditAuthor.getName().toLowerCase())))
                .collect(Collectors.toList());
        return new Page.Builder<AuditAuthor, String>(auditAuthors, page.getIsLastPage()).nextPageRequest(page.getNextPageRequest().orElse(null)).build();
    }

    @VisibleForTesting
    boolean authorMatches(String search, AuditAuthor auditAuthor) {
        if (auditAuthor.getName() == null) {
            return false;
        }
        final String searchStr = (search == null ? "" : search).toLowerCase();
        return Collections.list(new StringTokenizer(auditAuthor.getName().toLowerCase(), DELIMS)).stream()
                .map(str -> (String) str)
                .anyMatch(str -> str.startsWith(searchStr));
    }

}
