package com.atlassian.audit.frontend.servlet;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.analytics.ViewEvent;
import com.atlassian.audit.permission.PermissionChecker;
import com.atlassian.audit.plugin.AuditPluginInfo;
import com.atlassian.audit.rest.DelegatedViewTypeProvider;
import com.atlassian.audit.spi.feature.DelegatedViewFeature;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.auth.LoginUriProvider;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.user.UserProfile;
import com.atlassian.sal.api.user.UserRole;
import com.atlassian.sal.api.websudo.WebSudoManager;
import com.atlassian.sal.api.websudo.WebSudoSessionException;
import com.atlassian.soy.renderer.SoyException;
import com.atlassian.soy.renderer.SoyTemplateRenderer;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.atlassian.sal.api.ApplicationProperties.PLATFORM_JIRA;
import static java.util.Collections.singletonMap;

public class AuditServlet extends HttpServlet {

    private static final String ENCODING = StandardCharsets.UTF_8.name();
    @VisibleForTesting
    static final String RESOURCE_KEY = "com.atlassian.audit.atlassian-audit-plugin:audit-base-resources";
    private static final String TEMPLATE_KEY = "atlassian.audit.auditBase";
    @VisibleForTesting
    static final String TEMPLATE_KEY_UNAUTHORISED = "atlassian.audit.auditUnauthorised";
    // JIRA specific attribute to let Seraph know that it should re-show the login page
    @VisibleForTesting
    static final String JIRA_SERAPH_SECURITY_ORIGINAL_URL = "os_security_originalurl";

    private final LoginUriProvider loginUriProvider;
    private final SoyTemplateRenderer soyTemplateRenderer;
    private final UserManager userManager;
    private final ApplicationProperties applicationProperties;
    private final PermissionChecker permissionChecker;
    private final EventPublisher eventPublisher;
    private final AuditPluginInfo auditPluginInfo;
    private final DelegatedViewFeature delegatedViewFeature;
    private final WebSudoManager webSudoManager;

    public AuditServlet(final LoginUriProvider loginUriProvider,
                        final SoyTemplateRenderer soyTemplateRenderer,
                        final UserManager userManager,
                        final ApplicationProperties applicationProperties,
                        final EventPublisher eventPublisher,
                        final PermissionChecker permissionChecker,
                        final DelegatedViewFeature delegatedViewFeature,
                        final AuditPluginInfo auditPluginInfo,
                        final WebSudoManager webSudoManager
    ) {
        this.loginUriProvider = loginUriProvider;
        this.soyTemplateRenderer = soyTemplateRenderer;
        this.userManager = userManager;
        this.applicationProperties = applicationProperties;
        this.eventPublisher = eventPublisher;
        this.permissionChecker = permissionChecker;
        this.auditPluginInfo = auditPluginInfo;
        this.delegatedViewFeature = delegatedViewFeature;
        this.webSudoManager = webSudoManager;
    }

    /**
     * Soy templates need params to not be null if you want to print them, so convert any
     * params that need it to an empty string.
     */
    private String safeSoyParam(String param) {
        if (param == null) {
            return "";
        }
        return param;
    }

    /**
     * Look for any params that start with "meta." and extract them by splitting at the dot
     * and returning a map with the parts after the dot as the key. This allows arbitrarily named
     * parameters to be passed to the soy templates.
     */
    private Map<String, Object> getMetaParams(Map<String, String[]> params) {
        Map<String, Object> metaParams = new HashMap<>();
        params.forEach((key, val) -> {
            if (key.startsWith("meta.")) {
                String keyName = key.split("\\.", 2)[1];
                metaParams.put(keyName, safeSoyParam(val[0]));
            }
        });
        return metaParams;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        final UserProfile user = this.userManager.getRemoteUser(req);
        if (user == null) {
            redirectToLogin(req, resp);
            return;
        }
        resp.setContentType("text/html;charset=" + ENCODING); // Bitbucket requires a content type
        Map<String, Object> params = new HashMap<>();

        if (req.getPathInfo().contains("/resource")) {
            String affectedObject = "";
            String resourceType = "";
            String resourceId = "";
            Map<String, Object> metaParams = getMetaParams(req.getParameterMap());

            // Extracts the affected object used for the resource view from the path
            Matcher matcher = Pattern.compile("/resource/(.+)/*").matcher(req.getPathInfo());
            if (matcher.find()) {
                affectedObject = matcher.group(1);
            }

            if (affectedObject != null && affectedObject.contains(",")) {
                String[] affectedObjectParts = affectedObject.split(",\\s*");
                resourceType = affectedObjectParts[0];
                resourceId = affectedObjectParts[1];
            }

            params.put("affectedObject", safeSoyParam(affectedObject));
            params.put("resourceId", safeSoyParam(resourceId));
            params.put("productName", applicationProperties.getDisplayName());
            params.put("isResourceView", true);
            params.putAll(metaParams);

            // If we are on the /resource path then it's important we pass the resourceType
            // so that the template can determine which resource template to render.
            params.put("resourceType", safeSoyParam(resourceType));

            if (delegatedViewFeature.isEnabled() && permissionChecker.hasResourceAuditViewPermission(resourceType, resourceId)) {
                publishViewEvent(safeSoyParam(resourceType));
                renderView(resp, params);
                return;
            }
        } else {
            if (permissionChecker.hasUnrestrictedAuditViewPermission()) {
                try {
                    webSudoManager.willExecuteWebSudoRequest(req);
                    publishViewEvent(DelegatedViewTypeProvider.GLOBAL);
                    renderView(resp, params);
                    return;
                } catch (WebSudoSessionException wes) {
                    webSudoManager.enforceWebSudoProtection(req, resp);
                    return;
                }
            }
        }

        if (isJiraPlatform()) {
            redirectToElevatedPermissionsLogin(req, resp);
        } else {
            renderUnauthorizedView(resp);
        }
    }

    private void redirectToLogin(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
        final URI uri = getUri(request);
        response.sendRedirect(loginUriProvider.getLoginUri(uri).toASCIIString());
    }

    private boolean isJiraPlatform() {
        return PLATFORM_JIRA.equals(applicationProperties.getPlatformId());
    }

    private void redirectToElevatedPermissionsLogin(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
        final URI uri = getUri(request);
        request.getSession().setAttribute(JIRA_SERAPH_SECURITY_ORIGINAL_URL, uri.toASCIIString());
        response.sendRedirect(loginUriProvider.getLoginUriForRole(uri, UserRole.ADMIN).toASCIIString());
    }

    private URI getUri(final HttpServletRequest request) {
        final StringBuffer requestURL = request.getRequestURL();
        if (request.getQueryString() != null) {
            requestURL
                    .append("?")
                    .append(request.getQueryString());
        }
        return URI.create(requestURL.toString());
    }

    private void renderView(HttpServletResponse resp, Map<String, Object> params) throws IOException, ServletException {
        render(resp, TEMPLATE_KEY, params);
    }

    private void renderUnauthorizedView(HttpServletResponse resp) throws IOException, ServletException {
        render(resp, TEMPLATE_KEY_UNAUTHORISED, singletonMap("message", "Unauthorised access"));
    }

    private void render(HttpServletResponse resp, String template, Map<String, Object> soyData) throws IOException, ServletException {
        try {
            soyTemplateRenderer.render(resp.getWriter(), RESOURCE_KEY, template, soyData);
        } catch (SoyException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            }
            throw new ServletException(e);
        }
    }

    private void publishViewEvent(String resourceType) {
        eventPublisher.publish(new ViewEvent(resourceType, auditPluginInfo.getPluginVersion()));
    }
}
