package com.atlassian.audit.frontend.data;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.coverage.ProductLicenseChecker;
import com.atlassian.audit.file.CachingRetentionFileConfigService;
import com.atlassian.audit.plugin.configuration.PropertiesProvider;
import com.atlassian.audit.service.TranslationService;
import com.atlassian.json.marshal.Jsonable;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Properties;
import java.util.function.Function;

import static com.atlassian.audit.file.FileAuditConsumer.DEFAULT_AUDIT_FILE_DIR;
import static com.atlassian.audit.retention.SalAuditRetentionFileConfigService.DEFAULT_MAX_FILE_SIZE_IN_MB;
import static com.atlassian.audit.schedule.db.limit.DbLimiterScheduler.AUDIT_DB_LIMIT_ROWS_DEFAULT;
import static com.atlassian.audit.schedule.db.limit.DbLimiterScheduler.AUDIT_DB_LIMIT_ROWS_KEY;

/**
 * Data provider for audit settings view
 */
public class AuditSettingsViewDataProvider implements WebResourceDataProvider {

    private static final String PROP_FILE_PATH = AuditEventsViewDataProvider.PROP_FILE_PATH;

    private static final String PREFIX_KEY = "atlassian.audit.frontend";
    private static final String EXPORT_MAX_RECORDS_WARNING_KEY = PREFIX_KEY + ".export.warning.maxRecord";
    private static final String AREA_KEY = PREFIX_KEY + ".area";
    private static final String LICENSE_DC_KEY = ".dc";
    private static final String LICENSE_SERVER_KEY = ".server";

    //areas
    private static final String AREA_LABEL_KEY = ".label";
    private static final String AREA_DESC_KEY = ".description";

    //levels
    private static final String LEVELS_KEY = PREFIX_KEY + ".levels";
    private static final String LEVEL_OFF_LABEL_KEY = PREFIX_KEY + ".configuration.levelInformation.label.off";
    private static final String LEVEL_BASE_LABEL_KEY = PREFIX_KEY + ".configuration.levelInformation.label.base";
    private static final String LEVEL_ADVANCED_LABEL_KEY = PREFIX_KEY + ".configuration.levelInformation.label.advanced";
    private static final String LEVEL_FULL_LABEL_KEY = PREFIX_KEY + ".configuration.levelInformation.label.full";

    private static final String SPLIT = ",";

    private final TranslationService translationService;
    private final ApplicationProperties applicationProperties;
    private final ProductLicenseChecker licenseChecker;
    private final Properties auditSettings;
    private final ObjectMapper objectMapper;
    private final CachingRetentionFileConfigService cachingRetentionFileConfigService;
    private final PropertiesProvider propertiesProvider;

    public AuditSettingsViewDataProvider(ProductLicenseChecker licenseChecker,
                                         ApplicationProperties applicationProperties,
                                         TranslationService translationService,
                                         ObjectMapper objectMapper,
                                         CachingRetentionFileConfigService cachingRetentionFileConfigService,
                                         PropertiesProvider propertiesProvider) throws IOException {
        this(licenseChecker, applicationProperties,
                translationService, objectMapper, PROP_FILE_PATH, cachingRetentionFileConfigService, propertiesProvider);
    }

    @VisibleForTesting
    protected AuditSettingsViewDataProvider(
            ProductLicenseChecker licenseChecker,
            ApplicationProperties applicationProperties,
            TranslationService translationService,
            ObjectMapper objectMapper,
            String propFilePath,
            CachingRetentionFileConfigService cachingRetentionFileConfigService,
            PropertiesProvider propertiesProvider) throws IOException {
        this.applicationProperties = applicationProperties;
        this.translationService = translationService;
        this.licenseChecker = licenseChecker;
        this.objectMapper = objectMapper;
        this.cachingRetentionFileConfigService = cachingRetentionFileConfigService;
        this.propertiesProvider = propertiesProvider;
        try (final InputStream inputStream = this.getClass().getResourceAsStream(propFilePath)) {
            auditSettings = new Properties();
            auditSettings.load(inputStream);
        }
    }

    @Override
    public Jsonable get() {
        return writer -> {
            try {
                objectMapper.writeValue(writer, getData());
            } catch (Exception e) {
                throw new JsonMappingException(e.getMessage(), e);
            }
        };
    }

    private AuditSettingsViewData getData() {
        AuditSettingsViewData settingsViewData = new AuditSettingsViewData()
                .withFileLocation("<HOME_DIRECTORY>/" + DEFAULT_AUDIT_FILE_DIR)
                .withMaxExportRecordsWarning(Integer.parseInt(auditSettings.getProperty(EXPORT_MAX_RECORDS_WARNING_KEY, "10000")))
                .withMaxRecordsInDb(propertiesProvider.getInteger(AUDIT_DB_LIMIT_ROWS_KEY, AUDIT_DB_LIMIT_ROWS_DEFAULT))
                .withFileSizeLimitInMb(propertiesProvider.getInteger("plugin.audit.file.max.file.size", DEFAULT_MAX_FILE_SIZE_IN_MB))
                .withMaxFileCount(cachingRetentionFileConfigService.getConfig().getMaxFileCount());

        String[] areaKeys = auditSettings.getProperty(
                new StringBuilder(AREA_KEY)
                        .append(isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)
                        .append(".")
                        .append(getProductName()).toString()).split(SPLIT);
        for (String areaKey : areaKeys) {
            settingsViewData.area(new AuditSettingsViewData.ConfigurationArea(
                    areaKey,
                    getAreaI18nText(areaKey, AREA_LABEL_KEY),
                    getAreaI18nText(areaKey, AREA_DESC_KEY),
                    getAvailableLevels(areaKey)));
        }
        settingsViewData.levels(new AuditSettingsViewData.ConfigurationLevel(
                getI18nText(LEVEL_OFF_LABEL_KEY),
                getI18nText(LEVEL_BASE_LABEL_KEY),
                getI18nText(LEVEL_ADVANCED_LABEL_KEY),
                getI18nText(LEVEL_FULL_LABEL_KEY)));
        settingsViewData.withLogFileEnabled(isDcLicense() ||
                ApplicationProperties.PLATFORM_BITBUCKET.equals(getProductName()) ||
                ApplicationProperties.PLATFORM_STASH.equals(getProductName()));
        return settingsViewData;
    }

    private String getI18nText(String key) {
        return translationService.getSiteLocaleText(key);
    }

    /**
     * Get i18 text in below order:
     * 1. atlassian.audit.frontend.{license}.{app}.{area}.{label}
     * 2. atlassian.audit.frontend.{license}.{area}.{label}
     * 3. atlassian.audit.frontend.{area}.{label}
     *
     * @param area
     * @param label can be "label" or "desc"
     * @return
     */
    private String getAreaI18nText(String area, String label) {
        return returnFirstTranslation(this::getI18nText,
                new StringBuilder(PREFIX_KEY)
                        .append(isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)
                        .append(".")
                        .append(getProductName())
                        .append(".")
                        .append(area)
                        .append(label)
                        .toString(),
                new StringBuilder(PREFIX_KEY)
                        .append(isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)
                        .append(".")
                        .append(area)
                        .append(label)
                        .toString(),
                new StringBuilder(PREFIX_KEY)
                        .append(".")
                        .append(area)
                        .append(label)
                        .toString()
        );
    }

    /**
     * Resolve property from an array of property keys, return the 1st - most specific - found property value.
     *
     * @param func
     * @param propKeys
     * @return
     */
    private String returnFirstTranslation(Function<String, String> func, String... propKeys) {
        for (String propKey : propKeys) {
            String propValue = func.apply(propKey);
            if (propValue != null && !propValue.equals(propKey)) {
                return propValue;
            }
        }
        throw new RuntimeException("Not able to find property from : " + Arrays.toString(propKeys));
    }

    /**
     * Get available levels for each area from properties file in below order:
     * 1. atlassian.audit.frontend.{license}.{app}.{area}
     * 2. atlassian.audit.frontend.{license}.{app}
     * 3. atlassian.audit.frontend.{license}
     *
     * @param area
     * @return
     */
    private String[] getAvailableLevels(String area) {
        return returnFirstTranslation(auditSettings::getProperty,
                new StringBuilder(LEVELS_KEY)
                        .append(isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)
                        .append(".")
                        .append(getProductName())
                        .append(".")
                        .append(area)
                        .toString(),
                new StringBuilder(LEVELS_KEY)
                        .append(isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)
                        .append(".")
                        .append(getProductName())
                        .toString(),
                new StringBuilder(LEVELS_KEY)
                        .append(isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)
                        .toString()).split(SPLIT);
    }

    private boolean isDcLicense() {
        return !licenseChecker.isNotDcLicense();
    }

    private String getProductName() {
        return applicationProperties.getPlatformId();
    }

}
