package com.atlassian.audit.frontend.data;

import com.atlassian.audit.coverage.ProductLicenseChecker;
import com.atlassian.audit.plugin.onboarding.OnboardingSeenService;
import com.atlassian.json.marshal.Jsonable;
import com.atlassian.plugin.webresource.UrlMode;
import com.atlassian.plugin.webresource.WebResourceUrlProvider;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import org.codehaus.jackson.map.ObjectMapper;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import static java.util.Collections.emptyList;

public class AuditOnboardingDataProvider implements WebResourceDataProvider {

    private static final String FALLBACK_URL = "http://www.atlassian.com/";

    private static final String PROP_FILE_PATH = "/atlassian-audit-onboarding-settings.properties";

    private final OnboardingSeenService onboardingSeenService;

    private final I18nResolver resolver;

    private final ObjectMapper objectMapper;

    private final ProductLicenseChecker licenseChecker;

    private final WebResourceUrlProvider webResourceUrlProvider;

    private final Properties auditSettings;

    private final ApplicationProperties applicationProperties;

    public AuditOnboardingDataProvider(final OnboardingSeenService onboardingSeenService,
                                       final ObjectMapper objectMapper, final I18nResolver resolver,
                                       final ProductLicenseChecker licenseChecker,
                                       final WebResourceUrlProvider webResourceUrlProvider,
                                       final ApplicationProperties applicationProperties
    ) throws IOException {
        this(onboardingSeenService, objectMapper, resolver, licenseChecker, webResourceUrlProvider,
                applicationProperties, PROP_FILE_PATH);
    }

    protected AuditOnboardingDataProvider(final OnboardingSeenService onboardingSeenService,
                                          final ObjectMapper objectMapper, final I18nResolver resolver,
                                          final ProductLicenseChecker licenseChecker,
                                          final WebResourceUrlProvider webResourceUrlProvider,
                                          final ApplicationProperties applicationProperties,
                                          final String propFilePath
    ) throws IOException {
        this.onboardingSeenService = onboardingSeenService;
        this.objectMapper = objectMapper;
        this.resolver = resolver;
        this.licenseChecker = licenseChecker;
        this.webResourceUrlProvider = webResourceUrlProvider;
        this.applicationProperties = applicationProperties;
        try (final InputStream inputStream = this.getClass().getResourceAsStream(propFilePath)) {
            auditSettings = new Properties();
            auditSettings.load(inputStream);
        }
    }

    @Override
    public Jsonable get() {
        return writer -> objectMapper.writeValue(writer, getData());
    }

    private AuditOnboardingData getData() {

        if (!onboardingSeenService.shouldDisplay()) {
            return new AuditOnboardingData(emptyList());
        }

        final boolean isDc = licenseChecker.isDcLicenseOrExempt();
        return new AuditOnboardingData(getOnboardingContents().stream()
                .filter(data -> isDc || !data.isDcOnly())
                .collect(Collectors.toList()));
    }

    private List<AuditingOnboardingDisplayInfoData> getOnboardingContents() {
        return IntStream.rangeClosed(1, getMaximumTabs())
                .mapToObj(this::createAuditingContent)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
    }

    private Optional<String> getProperty(final String propertyName) {
        String translatedText = resolver.getText(propertyName);
        if (translatedText == null || translatedText.equals(propertyName)) {
            return Optional.empty();
        }
        return Optional.of(translatedText);
    }
    
    private String getProductNameForPropertiesFile() {
        final String productName = applicationProperties.getPlatformId();
        if (productName.equals(ApplicationProperties.PLATFORM_BITBUCKET) ||
                productName.equals(ApplicationProperties.PLATFORM_STASH)) {
            return ApplicationProperties.PLATFORM_BITBUCKET;
        }
        return productName;
    }

    private Optional<AuditingOnboardingDisplayInfoData> createAuditingContent(final int index) {
        final Optional<String> dcOnlyFeature = Optional.ofNullable(
                auditSettings.getProperty(String.format("atlassian.audit.onboarding.%d.support.dc.only", index)));
        final Optional<String> title =
                getProperty(String.format("atlassian.audit.onboarding.%d.title", index));
        final Optional<String> description =
                getProperty(String.format("atlassian.audit.onboarding.%d.description", index));
        final Optional<String> confirmButtonLabel =
                getProperty(String.format("atlassian.audit.onboarding.%d.confirm.btn.label", index));
        final Optional<String> learnMoreButtonLabel =
                getProperty(String.format("atlassian.audit.onboarding.%d.learn.btn.label", index));
        final Optional<String> imageUrl = Optional.ofNullable(
                auditSettings.getProperty(String.format("atlassian.audit.onboarding.%d.image", index)));
        final Optional<String> articleUrl = Optional.ofNullable(
                auditSettings.getProperty(String.format("atlassian.audit.onboarding.%d.article." +
                        getProductNameForPropertiesFile(), index), FALLBACK_URL));

        if (!dcOnlyFeature.isPresent() || !title.isPresent() || !description.isPresent() || !imageUrl.isPresent() ||
                !articleUrl.isPresent() || !confirmButtonLabel.isPresent() || !learnMoreButtonLabel.isPresent()
        ) {
            return Optional.empty();
        }

        final String image = webResourceUrlProvider
                .getStaticPluginResourceUrl("com.atlassian.audit.atlassian-audit-plugin:audit-resources",
                        imageUrl.get(), UrlMode.AUTO);
        return Optional.of(new AuditingOnboardingDisplayInfoData(Boolean.parseBoolean(dcOnlyFeature.get()), title.get(),
                description.get(), image, articleUrl.get(), confirmButtonLabel.get(), learnMoreButtonLabel.get()));
    }

    private int getMaximumTabs() {
        return Integer.parseInt(auditSettings.getProperty("atlassian.audit.onboarding.maximum.tabs"));
    }
}
