package com.atlassian.audit.frontend.data;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.json.marshal.Jsonable;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.Properties;

/**
 * Data provider for KB articles.
 */
public class AuditKbArticleDataProvider implements WebResourceDataProvider {

    private static final String PROP_FILE_PATH = AuditEventsViewDataProvider.PROP_FILE_PATH;

    private static final String PREFIX_KEY = "atlassian.audit.frontend";
    private static final String KB_FEATURE_KEY = PREFIX_KEY + ".kb.feature.";
    private static final String KB_REFERENCE_KEY = PREFIX_KEY + ".kb.reference.";
    private static final String KB_INTEGRATIONS_KEY = PREFIX_KEY + ".kb.integrations.";
    private static final String KB_DATABASE_RETENTION_KEY = PREFIX_KEY + ".kb.databaseRetention";

    private final ApplicationProperties applicationProperties;
    private final Properties auditSettings;
    private final ObjectMapper objectMapper;

    public AuditKbArticleDataProvider(
            ApplicationProperties applicationProperties,
            ObjectMapper objectMapper) throws IOException {
        this(applicationProperties, objectMapper, PROP_FILE_PATH);
    }

    @VisibleForTesting
    protected AuditKbArticleDataProvider(
            ApplicationProperties applicationProperties,
            ObjectMapper objectMapper,
            String propFilePath) throws IOException {
        this.applicationProperties = applicationProperties;
        this.objectMapper = objectMapper;
        try (final InputStream inputStream = this.getClass().getResourceAsStream(propFilePath)) {
            auditSettings = new Properties();
            auditSettings.load(inputStream);
        }
    }

    @Override
    public Jsonable get() {
        return writer -> {
            try {
                objectMapper.writeValue(writer, getData());
            } catch (Exception e) {
                throw new JsonMappingException(e.getMessage(), e);
            }
        };
    }

    private AuditKbArticleData getData() {
        String fallbackUrl = "http://www.atlassian.com/";
        return new AuditKbArticleData(
                URI.create(auditSettings.getProperty(KB_FEATURE_KEY + getProductName(), fallbackUrl)),
                URI.create(auditSettings.getProperty(KB_REFERENCE_KEY + getProductName(), fallbackUrl)),
                URI.create(auditSettings.getProperty(KB_INTEGRATIONS_KEY + getProductName(), fallbackUrl)),
                URI.create(auditSettings.getProperty(KB_DATABASE_RETENTION_KEY, fallbackUrl))); // product agnostic);

    }

    private String getProductName() {
        return applicationProperties.getPlatformId();
    }

}
