package com.atlassian.audit.frontend.data;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.audit.ao.service.CachedCountService;
import com.atlassian.audit.coverage.ProductLicenseChecker;
import com.atlassian.audit.search.ActionsProvider;
import com.atlassian.audit.search.CategoriesProvider;
import com.atlassian.audit.service.TranslationService;
import com.atlassian.json.marshal.Jsonable;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.timezone.TimeZoneManager;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import org.codehaus.jackson.map.JsonMappingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toMap;

/**
 * Data provider for audit events view.
 */
public class AuditEventsViewDataProvider implements WebResourceDataProvider {

    protected static final String PROP_FILE_PATH = "/atlassian-audit-frontend-settings.properties";

    private static final String PREFIX_KEY = "atlassian.audit.frontend";
    private static final String TITLE_KEY = PREFIX_KEY + ".title";
    private static final String LICENSE_DC_KEY = ".dc";
    private static final String LICENSE_SERVER_KEY = ".server";

    //Affected object filters
    private static final String AFFECTED_OBJECTS_PLACEHOLDER = PREFIX_KEY + ".affectedObjects.placeholder.";
    private static final String AFFECTED_OBJECTS_TYPE = PREFIX_KEY + ".affectedObjects.type.";
    private static final String DELEGATED_VIEW_FILTER = PREFIX_KEY + ".delegatedView.filter.";

    private static final String SPLIT = ",";

    private static final Logger log = LoggerFactory.getLogger(AuditEventsViewDataProvider.class);

    private final TranslationService translationService;
    private final ApplicationProperties applicationProperties;
    private final ProductLicenseChecker licenseChecker;
    private final Properties auditSettings;
    private final ObjectMapper objectMapper;
    private final TimeZoneManager timeZoneManager;
    private final CategoriesProvider categoriesProvider;
    private final ActionsProvider actionsProvider;
    private final CachedCountService cachedCountService;

    public AuditEventsViewDataProvider(ProductLicenseChecker licenseChecker,
                                       ApplicationProperties applicationProperties,
                                       TranslationService translationService,
                                       ObjectMapper objectMapper,
                                       TimeZoneManager timeZoneManager,
                                       CategoriesProvider categoriesProvider,
                                       ActionsProvider actionsProvider,
                                       CachedCountService cachedCountService) throws IOException {
        this(licenseChecker, applicationProperties,
                translationService, objectMapper,
                timeZoneManager, PROP_FILE_PATH, categoriesProvider, actionsProvider, cachedCountService);
    }

    @VisibleForTesting
    protected AuditEventsViewDataProvider(
            ProductLicenseChecker licenseChecker,
            ApplicationProperties applicationProperties,
            TranslationService translationService,
            ObjectMapper objectMapper,
            TimeZoneManager timeZoneManager,
            String propFilePath,
            CategoriesProvider categoriesProvider,
            ActionsProvider actionsProvider, 
            CachedCountService cachedCountService) throws IOException {
        this.applicationProperties = applicationProperties;
        this.translationService = translationService;
        this.licenseChecker = licenseChecker;
        this.objectMapper = objectMapper;
        this.timeZoneManager = timeZoneManager;
        this.actionsProvider = actionsProvider;
        try (final InputStream inputStream = this.getClass().getResourceAsStream(propFilePath)) {
            auditSettings = new Properties();
            auditSettings.load(inputStream);
        }
        this.categoriesProvider = categoriesProvider;
        this.cachedCountService = cachedCountService;
    }

    @Override
    public Jsonable get() {
        return writer -> {
            try {
                objectMapper.writeValue(writer, getData());
            } catch (Exception e) {
                throw new JsonMappingException(e.getMessage(), e);
            }
        };
    }

    private AuditEventsViewData getData() {
        AuditEventsViewData eventsViewData = new AuditEventsViewData()
                .withPageTitle(getI18nText(TITLE_KEY + (isDcLicense() ? LICENSE_DC_KEY : LICENSE_SERVER_KEY)));

        eventsViewData.withSelectiveExportEnabled(isDcLicense() ||
                ApplicationProperties.PLATFORM_CONFLUENCE.equals(getProductName()))
                .withUserLocale(getUserLocale())
                .withUserTimeZone(getUserTimeZone())
                .withServerTimeZone(getServerTimeZone())
                .withDcFilter(isDcLicense())
                .categoryFilter(getCategories())
                .actionFilter(getActions())
                .count(getCount());
        eventsViewData.affectedObjectsFilters(getAffectedObjectFiltersForProduct());
        eventsViewData.globalAffectedObjectsFilters(getGlobalAffectedObjectFiltersForProduct());
        eventsViewData.delegatedAffectedObjectsFilters(getDelegatedViewsAffectedObjectsFilters());

        return eventsViewData;
    }

    private long getCount() {
        return cachedCountService.count();
    }

    private List<String> getCategories() {
        try {
            return categoriesProvider.getCategories();
        } catch (RuntimeException e) {
            // when error occurs, we don't want to crash the UI, just show empty options for category filter
            log.error("Failed to get category filters", e);
            return Collections.emptyList();
        }
    }

    private List<String> getActions() {
        try {
            return actionsProvider.getActions();
        } catch (RuntimeException e) {
            log.error("Failed to get action filters", e);
            return Collections.emptyList();
        }
    }

    private String getI18nText(String key) {
        return translationService.getSiteLocaleText(key);
    }

    private String getServerTimeZone() {
        return timeZoneManager.getDefaultTimeZone().getID();
    }

    private String getUserTimeZone() {
        return timeZoneManager.getUserTimeZone().getID();
    }

    private String getUserLocale() {
        return translationService.getUserLocale().toLanguageTag();
    }

    private boolean isDcLicense() {
        return !licenseChecker.isNotDcLicense();
    }

    private String getProductName() {
        return applicationProperties.getPlatformId();
    }

    private Collection<AuditEventsViewData.AffectedObjectsFilter> getAffectedObjectFiltersForProduct() {
        String filtersStr = auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + getProductName(), "");
        return Stream.of(filtersStr.split(SPLIT))
                .filter(str -> str != null && str.length() > 0)
                .map(key ->
                        new AuditEventsViewData.AffectedObjectsFilter(
                                auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + key),
                                getI18nText(AFFECTED_OBJECTS_PLACEHOLDER + key)))
                .collect(Collectors.toList());
    }

    private Collection<String> getGlobalAffectedObjectFiltersForProduct() {
        String filtersStr = auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + getProductName(), "");
        return Stream.of(filtersStr.split(SPLIT))
                .filter(str -> str != null && str.length() > 0)
                .map(key -> auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + key))
                .collect(Collectors.toList());
    }

    private Map<String, List<String>> getDelegatedViewsAffectedObjectsFilters() {
        String filtersStr = auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + getProductName(), "");
        return Stream.of(filtersStr.split(SPLIT))
                .filter(str -> str != null && str.length() > 0)
                .collect(toMap(
                        key -> auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + key),
                        key -> Stream.of(auditSettings.getProperty(DELEGATED_VIEW_FILTER + getProductName() + "." + key, "").split(SPLIT))
                                .filter(str -> str != null && str.length() > 0)
                                .map(filter -> auditSettings.getProperty(AFFECTED_OBJECTS_TYPE + filter, ""))
                                .collect(Collectors.toList())));
    }
}
