package com.atlassian.audit.file;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Objects;
import java.util.function.Supplier;

import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.util.stream.Collectors.joining;

/**
 * A publisher for continuously writing to a given file managed by the {@link Supplier}.
 */
public class FileMessagePublisher {
    private static final Logger log = LoggerFactory.getLogger(FileMessagePublisher.class);

    private final Supplier<Path> currentAuditFilePathSupplier;

    public FileMessagePublisher(Supplier<Path> currentAuditFilePathSupplier) {
        this.currentAuditFilePathSupplier = currentAuditFilePathSupplier;
    }

    public synchronized void publish(@Nonnull String... messages) {
        Objects.requireNonNull(messages);
        Path currentPath = currentAuditFilePathSupplier.get();
        try (OutputStream out = Files.newOutputStream(currentPath, CREATE, APPEND)) {
            String data = Arrays.stream(messages)
                                .filter(this::isValidMessage)
                                .collect(joining(System.lineSeparator(), "", System.lineSeparator()));
            out.write(data.getBytes(StandardCharsets.UTF_8));
        } catch (Exception e) {
            log.error("Unable to write {} to {}", messages, currentPath, e);
        }
    }

    private boolean isValidMessage(@Nullable final String message) {
        if (message == null) {
            // allow empty "" messages for now because ... (shrug) ?
            log.debug("message is null. Not publishing.");
            return false;
        }
        return true;
    }
}
