package com.atlassian.audit.coverage;

import com.atlassian.sal.api.license.BaseLicenseDetails;
import com.atlassian.sal.api.license.LicenseHandler;

/**
 * This class is responsible to answer important questions about the product license
 */
public class ProductLicenseChecker {

    // License property used in Confluence tests to avoid having to set up a Data Center instance.
    // The reason why we need it - Confluence can't execute plugin integration tests with DC license.
    private static final String CONFLUENCE_BYPASS_DATA_CENTER_CHECK_PROPERTY_KEY =
            "confluence.bypass.data.center.check";

    private final LicenseHandler licenseHandler;

    public ProductLicenseChecker(LicenseHandler licenseHandler) {
        this.licenseHandler = licenseHandler;
    }

    public boolean isNotDcLicense() {
        return !isDcLicenseOrExempt();
    }

    public boolean isDcLicenseOrExempt() {
        return licenseHandler.getAllProductLicenses().stream().allMatch(this::isDcLicenseOrExempt);
    }

    /**
     * The hack with special license property is required because Confluence can't execute plugin integration tests
     * with DC license. The workaround - to use Server license with special property to test DC-only features.
     * <p>
     * After the underlying problem is fixed (Connie can test against DC), this check can be removed.
     * <a href="https://bulldog.internal.atlassian.com/browse/QUICK-1234">Ticket</a>
     */
    private boolean isDcLicenseOrExempt(BaseLicenseDetails license) {
        return "true".equals(license.getProperty(CONFLUENCE_BYPASS_DATA_CENTER_CHECK_PROPERTY_KEY)) ||
                license.isDataCenter();
    }

}
