package com.atlassian.audit.coverage;

import com.atlassian.audit.api.AuditCoverageConfigService;
import com.atlassian.audit.entity.AuditCoverageConfig;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * A wrapper around coverage configuration service which caches the current value.
 */
public class CachingAuditCoverageService implements InitializingBean, DisposableBean {

    private final SingleValueCache<AuditCoverageConfig> configCache;

    private final EventPublisher eventPublisher;

    public CachingAuditCoverageService(EventPublisher eventPublisher, AuditCoverageConfigService delegate, int expirationSeconds) {
        this.eventPublisher = eventPublisher;
        configCache = new SingleValueCache<>(delegate::getConfig, expirationSeconds, SECONDS);
    }

    public AuditCoverageConfig getConfig() {
        return configCache.get();
    }

    /**
     * Please keep in mind that we invalidate the cache only when coverage was changed on the local node.
     * If somebody changes coverage on another DC node, the cache will know nothing about it and expire after timeout.
     * This is fine to have the lag - as long as cache expiration timeout is not large.
     * We don't use cluster events here because there's no X-product way to do them.
     */
    @EventListener
    public void onCoverageUpdatedEvent(CoverageUpdatedEvent event) {
        configCache.invalidate();
    }

    @Override
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }

    @Override
    public void destroy() {
        eventPublisher.unregister(this);
    }
}
