package com.atlassian.audit.broker;

import com.atlassian.audit.api.AuditConsumer;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.plugin.configuration.PropertiesProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.joining;

public class LoggingAuditConsumerExceptionHandler implements AuditConsumerExceptionHandler {

    public static final Logger LOGGER = LoggerFactory.getLogger(LoggingAuditConsumerExceptionHandler.class);

    private static final String LOGGED_COUNT_THRESHOLD_KEY = "plugin.audit.broker.exception.loggedCount";
    private static final int LOGGED_COUNT_DEFAULT = 3;
    private final Logger log;
    private final int threshold;

    public LoggingAuditConsumerExceptionHandler(Logger log, PropertiesProvider propertiesProvider) {
        this.log = requireNonNull(log);
        this.threshold = propertiesProvider.getInteger(LOGGED_COUNT_THRESHOLD_KEY, LOGGED_COUNT_DEFAULT);
    }

    @Override
    public void handle(AuditConsumer auditConsumer, RuntimeException exception, List<AuditEntity> batch) {
        log.error("Error occurred in {} while processing {} events {}",
                auditConsumer,
                batch == null ? 0 : batch.size(),
                log.isDebugEnabled() ? toString(batch) : "",
                exception);
    }

    private String toString(List<AuditEntity> batch) {
        if (batch == null) {
            return null;
        }
        return batch.stream()
                .limit(threshold)
                .map(this::toString)
                .collect(joining("," + System.lineSeparator(),
                        "[" + System.lineSeparator(), "]"));
    }

    /**
     * Instead of using AuditEntity.toString(), this method convert AuditEntity to String without
     * affected objects
     * changed value
     * extra attributes
     * to avoid making logger file too verbose.
     *
     * @param auditEntity
     * @return
     */
    private String toString(AuditEntity auditEntity) {
        return auditEntity == null ? "null" :
                "AuditEntity{" +
                        "version='" + auditEntity.getVersion() + '\'' +
                        ", timestamp=" + auditEntity.getTimestamp() +
                        ", author=" + auditEntity.getAuthor() +
                        ", auditType='" + auditEntity.getAuditType() + '\'' +
                        ", source='" + auditEntity.getSource() + '\'' +
                        ", system='" + auditEntity.getSystem() + '\'' +
                        ", node='" + auditEntity.getNode() + '\'' +
                        ", method=" + auditEntity.getMethod() +
                        '}';
    }

}
