package com.atlassian.audit.broker;

import com.atlassian.audit.api.AuditConsumer;
import com.atlassian.audit.event.AuditConsumerAddedEvent;
import com.atlassian.audit.event.AuditConsumerRemovedEvent;
import com.atlassian.event.api.EventPublisher;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Nonnull;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import static java.util.Objects.requireNonNull;

public class AuditConsumerRegistry implements ServiceTrackerCustomizer<AuditConsumer, AuditConsumer>, InitializingBean, DisposableBean {

    private static final Logger log = LoggerFactory.getLogger(AuditConsumerRegistry.class);

    private final Set<AuditConsumer> consumers = Collections.newSetFromMap(new ConcurrentHashMap<>());

    private final EventPublisher eventPublisher;

    private final BundleContext bundleContext;

    private volatile ServiceTracker<AuditConsumer, AuditConsumer> serviceTracker;

    public AuditConsumerRegistry(@Nonnull EventPublisher eventPublisher,
                                 @Nonnull BundleContext bundleContext) {
        this.eventPublisher = requireNonNull(eventPublisher);
        this.bundleContext = requireNonNull(bundleContext);
    }

    /**
     * Method provided to allow products to register {@link AuditConsumer}s directly
     */
    public void registerConsumer(@Nonnull AuditConsumer consumer) {
        if (!consumers.add(consumer)) {
            log.debug("AuditConsumer of type {} has already been tracked",
                    consumer.getClass().getName());
            return;
        }

        eventPublisher.publish(new AuditConsumerAddedEvent(consumer));
    }

    /**
     * Method provided to allow products to register {@link AuditConsumer}s directly
     */
    public void removeConsumer(@Nonnull AuditConsumer consumer) {
        if (!consumers.remove(consumer)) {
            log.debug("Removed service AuditConsumer of type {} was not being tracked",
                    consumer.getClass().getName());
        }

        eventPublisher.publish(new AuditConsumerRemovedEvent(consumer));
    }

    public void startTrackingAndAddInitialConsumers() {
        serviceTracker = new ServiceTracker<>(bundleContext, AuditConsumer.class, this);
        serviceTracker.open();

        // Ensure to add any currently registered services
        try {
            bundleContext.getServiceReferences(AuditConsumer.class, null).forEach(this::addingService);
        } catch (InvalidSyntaxException e) {
            log.error("Failed to register AuditConsumer", e);
        }
    }

    @Override
    public AuditConsumer addingService(ServiceReference<AuditConsumer> serviceReference) {
        final AuditConsumer consumer = bundleContext.getService(serviceReference);

        if (consumer == null) {
            log.debug("Failed to resolve AuditConsumer from bundle {} for reference {}",
                    serviceReference.getBundle(), serviceReference);
            return null;
        }

        registerConsumer(consumer);
        return consumer;
    }

    @Override
    public void modifiedService(ServiceReference<AuditConsumer> serviceReference, AuditConsumer consumer) {
        // We don't care if consumer properties were changed.
    }

    @Override
    public void removedService(ServiceReference<AuditConsumer> serviceReference, AuditConsumer consumerService) {
        bundleContext.ungetService(serviceReference);
        removeConsumer(consumerService);
    }

    @Override
    public void afterPropertiesSet() {
        startTrackingAndAddInitialConsumers();
    }

    @Override
    public void destroy() {
        if (serviceTracker != null) {
            serviceTracker.close();
        }
    }
}