package com.atlassian.audit.ao.service;

import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.AuditSearchService;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.permission.PermissionChecker;
import com.atlassian.plugins.rest.common.security.AuthorisationException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

/**
 * A wrapper which fails if the user doesn't have a permission for searching audit events.
 */
public class RestrictiveSearchService implements AuditSearchService {

    private final PermissionChecker permissionChecker;

    private final AuditSearchService origin;

    public RestrictiveSearchService(PermissionChecker permissionChecker, AuditSearchService origin) {
        this.permissionChecker = permissionChecker;
        this.origin = origin;
    }

    @Nonnull
    @Override
    public Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query, @Nonnull PageRequest<AuditEntityCursor> pageRequest, int scanLimit) throws TimeoutException {
        if (!permitted(query)) {
            throw new AuthorisationException("The user is not allowed to view audit events");
        }

        return origin.findBy(query, pageRequest, scanLimit);
    }

    @Override
    public void stream(@Nonnull AuditQuery query, int offset, int limit, @Nonnull Consumer<AuditEntity> consumer) throws TimeoutException {
        if (!permitted(query)) {
            throw new AuthorisationException("The user is not allowed to view audit events");
        }

        origin.stream(query, offset, limit, consumer);

    }

    @Override
    public long count(@Nullable AuditQuery query) throws TimeoutException {
        if (!permitted(query)) {
            throw new AuthorisationException("The user is not allowed to view audit events");
        }

        return origin.count(query);
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private boolean permitted(@Nullable AuditQuery query) {
        if (permissionChecker.hasUnrestrictedAuditViewPermission()) {
            return true;
        }

        // Query is not restricted and the user doesn't have an access to full audit log
        if (query == null) {
            return false;
        }

        // Apply delegated view permissions. The user should be allowed to see all resources.
        if (query.getResources().size() > 0) {
            return query.getResources().stream()
                    .allMatch(r -> permissionChecker.hasResourceAuditViewPermission(r.getType(), r.getId()));
        }

        return false;
    }
}
