package com.atlassian.audit.ao.service;

import com.atlassian.audit.analytics.SearchEvent;
import com.atlassian.audit.analytics.SearchTimeoutEvent;
import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.AuditSearchService;
import com.atlassian.audit.api.util.pagination.Page;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.atlassian.audit.entity.AuditEntity;
import com.atlassian.audit.plugin.AuditPluginInfo;
import com.atlassian.audit.rest.DelegatedViewTypeProvider;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.sal.api.web.context.HttpContext;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

public class AnalyticsTrackedAuditSearchService implements AuditSearchService {

    private final AuditSearchService delegate;
    private final EventPublisher eventPublisher;
    private final AuditPluginInfo pluginInfo;
    private final HttpContext httpContextSupplier;
    private final DelegatedViewTypeProvider delegatedViewTypeProvider = new DelegatedViewTypeProvider();

    public AnalyticsTrackedAuditSearchService(AuditSearchService delegate, EventPublisher eventPublisher, AuditPluginInfo pluginInfo,
                                              HttpContext httpContextSupplier) {
        this.delegate = delegate;
        this.eventPublisher = eventPublisher;
        this.pluginInfo = pluginInfo;
        this.httpContextSupplier = httpContextSupplier;
    }

    @Nonnull
    @Override
    public Page<AuditEntity, AuditEntityCursor> findBy(@Nonnull AuditQuery query,
                                                       @Nonnull PageRequest<AuditEntityCursor> pageRequest,
                                                       int scanLimit) throws TimeoutException {
        try {
            Page<AuditEntity, AuditEntityCursor> page = delegate.findBy(query, pageRequest, scanLimit);
            eventPublisher.publish(new SearchEvent(
                    query.getFrom().isPresent() || query.getTo().isPresent(),
                    !query.getUserIds().isEmpty(),
                    !query.getResources().isEmpty(),
                    !query.getCategories().isEmpty(),
                    !query.getActions().isEmpty(),
                    query.getSearchText().isPresent(),
                    scanLimit < Integer.MAX_VALUE && page.getIsLastPage(),
                    scanLimit == Integer.MAX_VALUE,
                    delegatedViewTypeProvider.getDelegatedViewType(httpContextSupplier),
                    pluginInfo.getPluginVersion()));
            return page;
        } catch (TimeoutException e) {
            eventPublisher.publish(new SearchTimeoutEvent(query.getSearchText().isPresent(), scanLimit < Integer.MAX_VALUE,
                    pluginInfo.getPluginVersion()));
            throw e;
        }
    }

    @Override
    public void stream(@Nonnull AuditQuery query, int offset, int limit, @Nonnull Consumer<AuditEntity> consumer) throws TimeoutException {
        delegate.stream(query, offset, limit, consumer);
    }

    @Override
    public long count(@Nullable AuditQuery query) throws TimeoutException {
        return delegate.count(query);
    }
}

