package com.atlassian.audit.ao.dao;

import com.atlassian.audit.entity.AuditResource;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.map.ObjectMapper;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

public class JacksonAffectedObjectsSerializer implements AffectedObjectsSerializer {
    private final ObjectMapper objectMapper;

    public static class AuditResourceData {
        private final String name;

        private final String type;

        private final String uri;

        private final String id;

        @JsonCreator
        public AuditResourceData(@JsonProperty("name") String name,
                                 @JsonProperty("type") String type,
                                 @JsonProperty("uri") String uri,
                                 @JsonProperty("id") String id) {
            this.name = name;
            this.type = type;
            this.uri = uri;
            this.id = id;
        }

        public String getName() {
            return name;
        }

        public String getType() {
            return type;
        }

        public String getUri() {
            return uri;
        }

        public String getId() {
            return id;
        }
    }

    public JacksonAffectedObjectsSerializer(ObjectMapper objectMapper) {
        this.objectMapper = requireNonNull(objectMapper);
    }

    @Override
    public List<AuditResource> deserialize(@Nonnull String s) {
        try {
            return Stream.of(objectMapper.readValue(s, AuditResourceData[].class))
                    .map(x -> AuditResource
                            .builder(ofNullable(x.getName()).orElse(
                                    ofNullable(x.getId()).orElse("Unknown")), x.getType())
                            .uri(x.getUri())
                            .id(x.getId())
                            .build())
                    .collect(Collectors.toList());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String serialize(@Nonnull List<AuditResource> affectedObjects) {
        AuditResourceData[] values = affectedObjects.stream()
                .map(x -> new AuditResourceData(x.getName(), x.getType(), x.getUri(), x.getId()))
                .toArray(AuditResourceData[]::new);
        try {
            return objectMapper.writeValueAsString(values);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
