package com.atlassian.audit.ao.dao;

import com.atlassian.audit.ao.dao.entity.AoAuditEntity;
import com.atlassian.audit.api.AuditEntityCursor;
import com.atlassian.audit.api.AuditQuery;
import com.atlassian.audit.api.util.pagination.PageRequest;
import com.google.common.collect.ImmutableSet;
import net.java.ao.Query;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static com.atlassian.audit.ao.dao.AuditQueryMapper.WhereClause.between;
import static com.atlassian.audit.ao.dao.AuditQueryMapper.WhereClause.eq;
import static com.atlassian.audit.ao.dao.AuditQueryMapper.WhereClause.greaterThan;
import static com.atlassian.audit.ao.dao.AuditQueryMapper.WhereClause.in;
import static com.atlassian.audit.ao.dao.AuditQueryMapper.WhereClause.lessThan;
import static com.atlassian.audit.ao.dao.AuditQueryMapper.WhereClause.like;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.ACTION_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.AREA_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.ATTRIBUTES_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.CATEGORY_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.CHANGE_VALUES_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.ID_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.LEVEL_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.METHOD_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.NODE_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCES_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_ID_COLUMN_1;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_ID_COLUMN_2;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_ID_COLUMN_3;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_ID_COLUMN_4;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_ID_COLUMN_5;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_TYPE_COLUMN_1;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_TYPE_COLUMN_2;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_TYPE_COLUMN_3;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_TYPE_COLUMN_4;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.RESOURCE_TYPE_COLUMN_5;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.SOURCE_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.SYSTEM_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.TIMESTAMP_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.USER_ID_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.USER_NAME_COLUMN;
import static com.atlassian.audit.ao.dao.entity.AoAuditEntity.USER_TYPE_COLUMN;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toSet;

public class AuditQueryMapper {

    public Query map(@Nonnull AuditQuery auditQuery,
                     @Nonnull PageRequest<AuditEntityCursor> pageRequest) {
        requireNonNull(auditQuery, "auditQuery");
        requireNonNull(pageRequest, "pageRequest");
        WhereClause clause = buildWhereClause(auditQuery, pageRequest);
        return Query.select(getColumnNames())
                .where(clause.getClause(), clause.getParams())
                .limit(pageRequest.getLimit() + 1)
                .order(format("%s DESC, %s DESC", TIMESTAMP_COLUMN, ID_COLUMN))
                .offset(pageRequest.getOffset());
    }

    public Query map(@Nonnull AuditQuery auditQuery) {
        requireNonNull(auditQuery, "auditQuery");
        WhereClause clause = buildWhereClause(auditQuery, new PageRequest.Builder().build());
        return Query.select(getColumnNames())
                .where(clause.getClause(), clause.getParams());
    }

    private static String getColumnNames() {
        return String.join(",", Arrays.asList(ACTION_COLUMN,
                AREA_COLUMN,
                ATTRIBUTES_COLUMN,
                CATEGORY_COLUMN,
                CHANGE_VALUES_COLUMN,
                LEVEL_COLUMN,
                METHOD_COLUMN,
                SYSTEM_COLUMN,
                NODE_COLUMN,
                RESOURCES_COLUMN,
                SOURCE_COLUMN,
                TIMESTAMP_COLUMN,
                USER_ID_COLUMN,
                USER_NAME_COLUMN,
                USER_TYPE_COLUMN));
    }

    private WhereClause buildWhereClause(@Nonnull AuditQuery auditQuery, @Nonnull PageRequest<AuditEntityCursor> pageRequest) {
        WhereClauseBuilder whereClauseBuilder = WhereClause.builder();
        pageRequest.getCursor().ifPresent(cursor ->
                whereClauseBuilder
                        .and(WhereClause.builder()
                                .lessThan(TIMESTAMP_COLUMN, cursor.getTimestamp().toEpochMilli(), false)
                                .or(WhereClause.builder()
                                        .eq(TIMESTAMP_COLUMN, cursor.getTimestamp().toEpochMilli())
                                        .and(lessThan(ID_COLUMN, cursor.getId(), false))
                                        .build()
                                ).build())
        );
        whereClauseBuilder
                .and(in(ACTION_COLUMN, auditQuery.getActions()))
                .and(in(CATEGORY_COLUMN, auditQuery.getCategories()))
                .and(likeClause(auditQuery))
                .and(in(USER_ID_COLUMN, auditQuery.getUserIds()))
                .and(between(TIMESTAMP_COLUMN, auditQuery.getFrom().orElse(Instant.EPOCH).toEpochMilli(), auditQuery.getTo().orElse(Instant.now()).toEpochMilli()));

        buildWhereClauseForResources(auditQuery.getResources(), whereClauseBuilder);
        if (auditQuery.getMinId().isPresent()) {
            whereClauseBuilder.and(greaterThan(ID_COLUMN, auditQuery.getMinId().get(), true));
        }
        if (auditQuery.getMaxId().isPresent()) {
            whereClauseBuilder.and(lessThan(ID_COLUMN, auditQuery.getMaxId().get(), true));
        }
        return whereClauseBuilder.build();
    }

    /**
     *  Build where clause by given AuditResourceIdentifier:
     *  <ul>
     *    <li>AND between different resource types</li>
     *    <li>OR with the same resource type</li>
     *  </ul>
     *
     *  see {@link AuditQuery#getResources}
     *
     */
    private void buildWhereClauseForResources(@Nonnull Set<AuditQuery.AuditResourceIdentifier> resources, WhereClauseBuilder whereClauseBuilder) {
        Map<String, Set<AuditQuery.AuditResourceIdentifier>> typeToAuditResourceIdentifierSetMap =
                resources.stream().collect(Collectors.groupingBy(AuditQuery.AuditResourceIdentifier::getType, toSet()));
        typeToAuditResourceIdentifierSetMap.values().forEach(auditResourceIdentifierSet -> whereClauseBuilder.and(WhereClause.builder()
                .or(buildWhereClauseForResourceColumn(auditResourceIdentifierSet,
                        RESOURCE_ID_COLUMN_1, RESOURCE_TYPE_COLUMN_1))
                .or(buildWhereClauseForResourceColumn(auditResourceIdentifierSet,
                        RESOURCE_ID_COLUMN_2, RESOURCE_TYPE_COLUMN_2))
                .or(buildWhereClauseForResourceColumn(auditResourceIdentifierSet,
                        RESOURCE_ID_COLUMN_3, RESOURCE_TYPE_COLUMN_3))
                .or(buildWhereClauseForResourceColumn(auditResourceIdentifierSet,
                        RESOURCE_ID_COLUMN_4, RESOURCE_TYPE_COLUMN_4))
                .or(buildWhereClauseForResourceColumn(auditResourceIdentifierSet,
                        RESOURCE_ID_COLUMN_5, RESOURCE_TYPE_COLUMN_5))
                .build()));
    }

    private WhereClause buildWhereClauseForResourceColumn(
            @Nonnull Set<AuditQuery.AuditResourceIdentifier> resourceIdentifiers,
            @Nonnull String resourceIdColumn,
            @Nonnull String resourceTypeColumn) {
        final WhereClauseBuilder builder = WhereClause.builder();
        resourceIdentifiers.forEach(r ->
                builder.or(WhereClause.builder().
                        eq(resourceIdColumn, r.getId())
                        .and(eq(resourceTypeColumn, r.getType()))
                        .build()));

        return builder.build();
    }

    private WhereClause likeClause(@Nonnull AuditQuery auditQuery) {
        ImmutableSet<String> tokens = new SearchTokenizer()
                .put(auditQuery.getSearchText().orElse(null))
                .getTokens();
        WhereClauseBuilder likeClause = WhereClause.builder();
        tokens.forEach(t -> likeClause.or(like(AoAuditEntity.SEARCH_STRING_COLUMN, "%" + t + "%")));
        return likeClause.build();
    }

    static class WhereClause {
        private final String clause;
        private final List<Object> params;

        private WhereClause(String clause, List<Object> params) {
            this.clause = clause;
            this.params = params;
        }

        static WhereClauseBuilder builder() {
            return new WhereClauseBuilder();
        }

        String getClause() {
            return clause;
        }

        Object[] getParams() {
            return params.toArray(new Object[0]);
        }

        boolean isEmpty() {
            return clause.isEmpty();
        }

        public static <T> WhereClause eq(@Nonnull String column, @Nullable T value) {
            return builder().eq(column, value).build();
        }

        @Nonnull
        public static <T> WhereClause in(@Nonnull String column, @Nullable Set<T> values) {
            return builder().in(column, values).build();
        }

        @Nonnull
        public static WhereClause like(@Nonnull String column, @Nullable String value) {
            return builder().like(column, value).build();
        }

        @Nonnull
        public static WhereClause between(@Nonnull String column, long value1, long value2) {
            return builder().between(column, value1, value2).build();
        }

        @Nonnull
        public static WhereClause greaterThan(@Nonnull String column, long value, boolean inclusive) {
            return builder().greaterThan(column, value, inclusive).build();
        }

        @Nonnull
        public static WhereClause lessThan(@Nonnull String column, long value, boolean inclusive) {
            return builder().lessThan(column, value, inclusive).build();
        }
    }

    private static class WhereClauseBuilder {
        private StringBuilder clause;
        private List<Object> params;

        private WhereClauseBuilder() {
            clause = new StringBuilder();
            params = new ArrayList<>();
        }

        WhereClause build() {
            return new WhereClause(clause.toString(), params);
        }

        @Nonnull
        WhereClauseBuilder and(@Nonnull WhereClause subClause) {
            requireNonNull(subClause, "subClause");
            if (!subClause.isEmpty()) {
                mayAppendAnd();
                clause.append(" (").append(subClause.getClause()).append(") ");
                params.addAll(Arrays.asList(subClause.getParams()));
            }
            return this;
        }

        @Nonnull
        WhereClauseBuilder or(@Nonnull WhereClause subClause) {
            requireNonNull(subClause, "subClause");
            if (!subClause.isEmpty()) {
                mayAppendOr();
                clause.append(" (").append(subClause.getClause()).append(") ");
                params.addAll(Arrays.asList(subClause.getParams()));
            }
            return this;
        }

        @Nonnull
        <T> WhereClauseBuilder eq(@Nonnull String column, @Nullable T value) {
            requireNonNull(column, "column");
            if (value != null) {
                append(column, value);
            }
            return this;
        }

        @Nonnull
        <T> WhereClauseBuilder in(@Nonnull String column, @Nullable Set<T> values) {
            requireNonNull(column, "column");
            requireNonNull(values, "values");
            if (!values.isEmpty()) {
                append(column, values);
            }
            return this;
        }

        @Nonnull
        WhereClauseBuilder like(@Nonnull String column, @Nullable String value) {
            requireNonNull(column, "column");
            if (value != null) {
                clause.append(column).append(" LIKE ?");
                params.add(value);
            }
            return this;
        }

        @Nonnull
        WhereClauseBuilder between(@Nonnull String column, long value1, long value2) {
            requireNonNull(column, "column");
            clause.append(column).append(" BETWEEN ? AND ?");
            params.add(value1);
            params.add(value2);
            return this;
        }

        @Nonnull
        WhereClauseBuilder greaterThan(@Nonnull String column, long value, boolean inclusive) {
            requireNonNull(column, "column");
            clause.append(column);
            if (inclusive) {
                clause.append(" >= ?");
            } else {
                clause.append(" > ?");
            }
            params.add(value);
            return this;
        }

        @Nonnull
        WhereClauseBuilder lessThan(@Nonnull String column, long value, boolean inclusive) {
            requireNonNull(column, "column");
            clause.append(column);
            if (inclusive) {
                clause.append(" <= ?");
            } else {
                clause.append(" < ?");
            }
            params.add(value);
            return this;
        }

        private void mayAppendAnd() {
            if (!params.isEmpty()) {
                clause.append(" AND ");
            }
        }

        private void mayAppendOr() {
            if (!params.isEmpty()) {
                clause.append(" OR ");
            }
        }

        private <T> void append(String column, T value) {
            clause.append(column).append(" = ?");
            params.add(value);
        }

        private <T> void append(String column, Set<T> values) {
            String collect = values.stream().map(i -> "?").collect(Collectors.joining(","));
            clause.append(column).append(" IN (").append(collect).append(")");
            params.addAll(values);
        }
    }
}
