package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * A name value pair represents extra info of an auditing entity.
 */
public class AuditAttribute {

    private final String name;

    private final String nameI18nKey;

    private final String value;

    /**
     * @deprecated since 1.7.0 use {@link #fromI18nKeys(String, String)} and provide the name i18n key.
     */
    @Deprecated
    public AuditAttribute(String name, String value) {
        this.nameI18nKey = name;
        this.name = name;
        this.value = value;
    }

    private AuditAttribute(Builder builder) {
        this.nameI18nKey = requireNonNull(builder.nameI18nKey);
        this.name = builder.name;
        this.value = builder.value;
    }

    /**
     * Audit Attribute name i18n key and value.
     * Example - for Audit Attribute "Extras" we expect input "audit.product.attribute.extras" for the name key which we
     * will then translate before displaying to user, value will not be translated.
     */
    public static Builder fromI18nKeys(@Nonnull String nameI18nKey, String value) {
        return new Builder(nameI18nKey, value);
    }

    /**
     * The attribute name - translation of {@link #getNameI18nKey()} by the auditing framework.
     */
    @Nullable
    public String getName() {
        return name;
    }

    /**
     * The attribute name i18n key. Expected unique for an attribute type and not expect to change.
     * Eg - for Audit Attribute "Extras" - "audit.product.myplugin.attribute.extras" which would then be used for
     * translation purposes.
     */
    @Nonnull
    public String getNameI18nKey() {
        return nameI18nKey;
    }

    @Nullable
    public String getValue() {
        return value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditAttribute that = (AuditAttribute) o;
        return Objects.equals(getName(), that.getName()) &&
                Objects.equals(getNameI18nKey(), that.getNameI18nKey()) &&
                Objects.equals(getValue(), that.getValue());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getName(), getNameI18nKey(), getValue());
    }

    @Override
    public String toString() {
        return "AuditAttribute{" +
                "name='" + name + '\'' +
                ", nameI18nKey='" + nameI18nKey + '\'' +
                ", value='" + value + '\'' +
                '}';
    }

    public static class Builder {

        private String name;
        private String nameI18nKey;
        private String value;

        public Builder(@Nonnull String nameI18nKey, String value) {
            this.nameI18nKey = requireNonNull(nameI18nKey);
            this.value = value;
        }

        public Builder withNameTranslation(String name) {
            this.name = name;
            return this;
        }

        public AuditAttribute build() {
            return new AuditAttribute(this);
        }
    }
}
