package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;

/**
 * Represents domain specific information of audit entity. Other fields of {@link AuditEntity} will be determined
 * automatically by the system.
 *
 * @since 1.1.0
 */
public class AuditEvent {

    @Nullable
    private final String action;

    @Nonnull
    private final String actionI18nKey;

    @Nullable
    private final String category;

    @Nonnull
    private final String categoryI18nKey;

    @Nonnull
    private final CoverageLevel level;

    @Nullable
    private final CoverageArea area;

    @Nonnull
    private final List<AuditResource> affectedObjects;

    @Nonnull
    private final List<ChangedValue> changedValues;

    @Nonnull
    private final Set<AuditAttribute> extraAttributes;

    private AuditEvent(Builder builder) {
        this.action = builder.action;
        this.actionI18nKey = requireNonNull(builder.actionI18nKey);
        this.category = builder.category;
        this.categoryI18nKey = requireNonNull(builder.categoryI18nKey);
        this.level = requireNonNull(builder.level);
        this.area = builder.area;
        this.affectedObjects = unmodifiableList(builder.affectedObjects);
        this.changedValues = unmodifiableList(builder.changedValues);
        this.extraAttributes = unmodifiableSet(builder.extraAttributes);
    }

    /**
     * @return the i18n key of scenario of the event, e.g. audit.product.action.projects.create.
     * @since 1.7.0
     */
    @Nonnull
    public String getActionI18nKey() {
        return actionI18nKey;
    }

    /**
     * @return the scenario of the event, e.g. create issue.
     * @deprecated since 1.7.0, see {@link #getActionI18nKey()}
     */
    @Nullable
    @Deprecated
    public String getAction() {
        return action;
    }

    /**
     * @return the Category i18n key.
     * @since 1.7.0
     * Example - for category "Pull Requests" we return "audit.product.category.pr"
     */
    @Nonnull
    public String getCategoryI18nKey() {
        return categoryI18nKey;
    }

    /**
     * @return the category of the event eg "Pull Requests"
     * @deprecated since 1.7.0, see {@link #getCategoryI18nKey()}
     */
    @Nullable
    @Deprecated
    public String getCategory() {
        return category;
    }

    @Nullable
    public CoverageArea getArea() {
        return area;
    }

    /**
     * @return estimated verbosity of the event
     */
    @Nonnull
    public CoverageLevel getLevel() {
        return level;
    }

    /**
     * @return the objects affected due to the event.
     */
    @Nonnull
    public List<AuditResource> getAffectedObjects() {
        return affectedObjects;
    }

    /**
     * @return the values changed due to the event.
     */
    @Nonnull
    public List<ChangedValue> getChangedValues() {
        return changedValues;
    }

    /**
     * @return extra details of the events, not all events have such attributes.
     */
    @Nonnull
    public Collection<AuditAttribute> getExtraAttributes() {
        return extraAttributes;
    }

    /**
     * @return get a specific attribute from the entity.
     */
    public Optional<String> getExtraAttribute(@Nonnull String name) {
        return extraAttributes.stream()
                .filter(a -> requireNonNull(name).equals(a.getName()))
                .findFirst()
                .map(AuditAttribute::getValue);
    }

    /**
     * @return get a specific attribute from the entity via nameKey.
     * @since 1.7.0
     */
    public Optional<String> getExtraAttributeByKey(@Nonnull String nameKey) {
        return extraAttributes.stream()
                .filter(a -> requireNonNull(nameKey).equals(a.getNameI18nKey()))
                .findFirst()
                .map(AuditAttribute::getValue);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuditEvent that = (AuditEvent) o;
        return Objects.equals(getAction(), that.getAction()) &&
                Objects.equals(getActionI18nKey(), that.getActionI18nKey()) &&
                Objects.equals(getCategory(), that.getCategory()) &&
                Objects.equals(getCategoryI18nKey(), that.getCategoryI18nKey()) &&
                getLevel() == that.getLevel() &&
                getArea() == that.getArea() &&
                Objects.equals(getAffectedObjects(), that.getAffectedObjects()) &&
                Objects.equals(getChangedValues(), that.getChangedValues()) &&
                Objects.equals(getExtraAttributes(), that.getExtraAttributes());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getAction(), getActionI18nKey(), getCategory(), getCategoryI18nKey(), getLevel(), getArea(), getAffectedObjects(), getChangedValues(), getExtraAttributes());
    }

    @Override
    public String toString() {
        return "AuditEvent{" +
                "action='" + action + '\'' +
                ", actionI18nKey='" + actionI18nKey + '\'' +
                ", category='" + category + '\'' +
                ", categoryI18nKey='" + categoryI18nKey + '\'' +
                ", level=" + level +
                ", area=" + area +
                ", affectedObjects=" + affectedObjects +
                ", changedValues=" + changedValues +
                ", extraAttributes=" + extraAttributes +
                '}';
    }

    /**
     * @deprecated since 1.7.0, use a variant with i18n keys {@link #fromI18nKeys(String, String, CoverageLevel)} or
     * {@link #fromI18nKeys(String, String, CoverageLevel, CoverageArea)}
     */
    @Deprecated
    public static Builder builder(@Nonnull String action, @Nonnull String category, @Nonnull CoverageLevel level) {
        return new Builder(action, category, level);
    }

    /**
     * @since 1.7.0
     */
    public static Builder fromI18nKeys(@Nonnull String categoryI18nKey, @Nonnull String actionI18nKey, @Nonnull CoverageLevel level, @Nullable CoverageArea area) {
        return new Builder(level, actionI18nKey, categoryI18nKey, area);
    }

    /**
     * @since 1.7.0
     */
    public static Builder fromI18nKeys(@Nonnull String categoryI18nKey, @Nonnull String actionI18nKey, @Nonnull CoverageLevel level) {
        return new Builder(level, actionI18nKey, categoryI18nKey, null);
    }

    public static Builder builder(@Nonnull AuditType type) {
        return new Builder(type);
    }

    /**
     * The Builder class for {@link AuditEvent}
     */
    public static class Builder {

        @Nonnull
        private String actionI18nKey;

        @Nullable
        private String action;

        @Nonnull
        private String categoryI18nKey;

        @Nullable
        private String category;

        @Nonnull
        private CoverageLevel level;

        @Nullable
        private CoverageArea area;

        @Nonnull
        private List<AuditResource> affectedObjects = new ArrayList<>();

        @Nonnull
        private List<ChangedValue> changedValues = new ArrayList<>();

        @Nonnull
        private Set<AuditAttribute> extraAttributes = new HashSet<>();

        public Builder(@Nonnull AuditType type) {
            this.actionI18nKey = requireNonNull(type.getActionI18nKey(), "actionI18nKey");
            this.action = type.getAction();
            this.categoryI18nKey = requireNonNull(type.getCategoryI18nKey(), "categoryI18nKey");
            this.category = type.getCategory();
            this.level = requireNonNull(type.getLevel(), "level");
            this.area = type.getArea();
        }

        private Builder(@Nonnull CoverageLevel level, @Nonnull String actionI18nKey, @Nonnull String categoryI18nKey, @Nullable CoverageArea area) {
            this.actionI18nKey = requireNonNull(actionI18nKey, "actionI18nKey");
            this.categoryI18nKey = requireNonNull(categoryI18nKey, "categoryI18nKey");
            this.level = requireNonNull(level, "level");
            this.area = area;
        }

        /**
         * @deprecated since 1.7.0, use a variant with i18n keys
         */
        @Deprecated
        public Builder(@Nonnull String action, @Nonnull String category, @Nonnull CoverageLevel level) {
            this(action, category, level, null);
        }

        /**
         * @deprecated since 1.7.0, use a variant with i18n keys
         */
        @Deprecated
        public Builder(@Nonnull String action, @Nonnull String category, @Nonnull CoverageLevel level, @Nullable CoverageArea area) {
            this.action = requireNonNull(action, "action");
            this.actionI18nKey = action;
            this.category = requireNonNull(category, "category");
            this.categoryI18nKey = category;
            this.level = requireNonNull(level, "level");
            this.area = area;
        }

        public Builder(@Nonnull AuditEvent event) {
            this.action = event.action;
            this.actionI18nKey = event.actionI18nKey;
            this.category = event.category;
            this.categoryI18nKey = event.categoryI18nKey;
            this.level = event.level;
            this.area = event.area;
            this.affectedObjects = new ArrayList<>(event.affectedObjects);
            this.changedValues = new ArrayList<>(event.changedValues);
            this.extraAttributes = new HashSet<>(event.extraAttributes);
        }

        /**
         * Action - translation of actionI18nKey
         * Example - for action "Pull Request Created" we expect input "Pull Request Created".
         * Not stored.
         *
         * @deprecated since 1.7.0 use {@link #actionI18nKey(String)} and provide the action i18n key.
         */
        @Deprecated
        public Builder action(@Nonnull String action) {
            this.action = requireNonNull(action);
            return this;
        }

        /**
         * Action i18n key.
         * Example - for action "Pull Request Created" we expect input "audit.product.action.pr.created" which we will then
         * translate before displaying to user.
         */
        public Builder actionI18nKey(@Nonnull String actionI18nKey) {
            this.actionI18nKey = requireNonNull(actionI18nKey);
            return this;
        }

        /**
         * Category i18n key.
         * Example - for category "Pull Requests" we expect input "audit.product.category.pr" which we will then
         * translate before displaying to user.
         */
        public Builder categoryI18nKey(@Nonnull String categoryI18nKey) {
            this.categoryI18nKey = requireNonNull(categoryI18nKey);
            return this;
        }

        /**
         * Category - translation of categoryI18nKey
         * Example - for category "Pull Requests" we expect input "Pull Request". Not stored.
         *
         * @deprecated since 1.7.0 use {@link #categoryI18nKey(String)} and provide the category i18n key.
         */
        @Deprecated
        public Builder category(@Nonnull String category) {
            this.category = requireNonNull(category);
            return this;
        }

        public Builder level(@Nonnull CoverageLevel level) {
            this.level = requireNonNull(level);
            return this;
        }

        public Builder area(@Nonnull CoverageArea area) {
            this.area = requireNonNull(area);
            return this;
        }

        public Builder affectedObjects(@Nonnull List<AuditResource> affectedObjects) {
            this.affectedObjects = requireNonNull(affectedObjects);
            return this;
        }

        public Builder appendAffectedObjects(@Nonnull List<AuditResource> affectedObjects) {
            this.affectedObjects.addAll(requireNonNull(affectedObjects));
            return this;
        }

        public Builder affectedObject(@Nonnull AuditResource affectedObject) {
            affectedObjects.add(requireNonNull(affectedObject));
            return this;
        }

        public Builder changedValues(@Nonnull List<ChangedValue> changedValues) {
            this.changedValues = requireNonNull(changedValues);
            return this;
        }

        public Builder appendChangedValues(@Nonnull Collection<ChangedValue> changedValues) {
            this.changedValues.addAll(requireNonNull(changedValues));
            return this;
        }

        public Builder changedValue(@Nonnull ChangedValue changedValue) {
            changedValues.add(requireNonNull(changedValue));
            return this;
        }

        /**
         * Adds {@link ChangedValue} only if the value changed.
         *
         * @param changedValue New {@link ChangedValue} to be added.
         * @return {@link Builder} instance
         * @since 1.1.0
         */
        public Builder addChangedValueIfDifferent(@Nonnull ChangedValue changedValue) {
            if (!Objects.equals(changedValue.getFrom(), changedValue.getTo())) {
                changedValue(changedValue);
            }
            return this;
        }

        public Builder extraAttributes(@Nonnull Collection<AuditAttribute> extraAttributes) {
            this.extraAttributes = new HashSet<>(requireNonNull(extraAttributes));
            return this;
        }

        public Builder appendExtraAttributes(@Nonnull Collection<AuditAttribute> extraAttributes) {
            this.extraAttributes.addAll(requireNonNull(extraAttributes));
            return this;
        }

        public Builder extraAttribute(@Nonnull AuditAttribute extraAttribute) {
            extraAttributes.add(requireNonNull(extraAttribute));
            return this;
        }

        public AuditEvent build() {
            return new AuditEvent(this);
        }
    }
}
