package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static java.time.temporal.ChronoUnit.MILLIS;
import static java.util.Collections.unmodifiableList;
import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;

/**
 * Represents an Auditing Entity.
 */
public class AuditEntity {

    private static final String version = "1.0";

    private final Long id;

    private final Instant timestamp;

    private final AuditAuthor author;

    private final AuditType auditType;

    private final List<AuditResource> affectedObjects;

    private final List<ChangedValue> changedValues;

    private final String source;

    private final String system;

    private final String node;

    private final String method;

    private final Set<AuditAttribute> extraAttributes;

    private AuditEntity(Builder builder) {
        this.id = builder.id;
        this.timestamp = builder.timestamp;
        this.author = builder.author;
        this.auditType = builder.type;
        this.affectedObjects = unmodifiableList(builder.affectedObjects);
        this.changedValues = unmodifiableList(builder.changedValues);
        this.source = builder.source;
        this.system = builder.system;
        this.node = builder.node;
        this.method = builder.method;
        this.extraAttributes = unmodifiableSet(builder.extraAttributes);
    }

    /**
     * @return Unique id of event. The id will be assigned by the backing storage and thus will only be returned when
     * retrieving or searching stored {@link AuditEntity} objects
     */
    @Nullable
    public Long getId() {
        return id;
    }

    /**
     * @return version of event type.
     */
    @Nonnull
    public String getVersion() {
        return version;
    }

    /**
     * @return when the event occurs in millisecond precision.
     */
    @Nonnull
    public Instant getTimestamp() {
        return timestamp;
    }

    /**
     * @return the author who initiated the event.
     */
    @Nonnull
    public AuditAuthor getAuthor() {
        return author;
    }

    /**
     * @return the scenario and category of the event, e.g. create issue.
     */
    @Nonnull
    public AuditType getAuditType() {
        return auditType;
    }

    /**
     * @return the objects affected due to the event.
     */
    @Nonnull
    public List<AuditResource> getAffectedObjects() {
        return affectedObjects;
    }

    /**
     * @return the values changed due to the event.
     */
    @Nonnull
    public List<ChangedValue> getChangedValues() {
        return changedValues;
    }

    /**
     * @return the client's IP address that triggered this event.
     */
    @Nullable
    public String getSource() {
        return source;
    }

    /**
     * @return the product instance where the event is generated, e.g. base url of Jira.
     */
    @Nullable
    public String getSystem() {
        return system;
    }

    /**
     * @return the node id where the event is generated, e.g. an node id in a Jira DC cluster.
     */
    @Nullable
    public String getNode() {
        return node;
    }

    /**
     * @return how the event occurred, e.g. Browser or Mobile or System or Unknown
     */
    @Nullable
    public String getMethod() {
        return method;
    }

    /**
     * @return extra details of the events, not all events have such attributes.
     */
    @Nonnull
    public Collection<AuditAttribute> getExtraAttributes() {
        return extraAttributes;
    }

    /**
     * @param name - the attribute name
     * @return get a specific attribute value from the entity found by name.
     */
    public Optional<String> getExtraAttribute(@Nonnull String name) {
        requireNonNull(name);
        return extraAttributes.stream()
                .filter(a -> name.equals(a.getName()))
                .findFirst()
                .map(AuditAttribute::getValue);
    }

    /**
     * @param i18nKey - the attribute name i18n key
     * @return get a specific attribute value from the entity found by i18nKey.
     */
    public Optional<String> getExtraAttributeByI18nKey(@Nonnull String i18nKey) {
        requireNonNull(i18nKey);
        return extraAttributes.stream()
                .filter(a -> i18nKey.equals(a.getNameI18nKey()))
                .findFirst()
                .map(AuditAttribute::getValue);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        AuditEntity that = (AuditEntity) o;
        return Objects.equals(getId(), that.getId()) &&
                Objects.equals(getTimestamp(), that.getTimestamp()) &&
                Objects.equals(getAuthor(), that.getAuthor()) &&
                Objects.equals(getAuditType(), that.getAuditType()) &&
                Objects.equals(getAffectedObjects(), that.getAffectedObjects()) &&
                Objects.equals(getChangedValues(), that.getChangedValues()) &&
                Objects.equals(getSource(), that.getSource()) &&
                Objects.equals(getSystem(), that.getSystem()) &&
                Objects.equals(getNode(), that.getNode()) &&
                Objects.equals(getMethod(), that.getMethod()) &&
                Objects.equals(getExtraAttributes(), that.getExtraAttributes());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getId(), getTimestamp(), getAuthor(), getAuditType(), getAffectedObjects(), getChangedValues(), getSource(), getSystem(), getNode(), getMethod(), getExtraAttributes());
    }

    @Override
    public String toString() {
        return "AuditEntity{" +
                "id ='" + id + '\'' +
                ", version='" + version + '\'' +
                ", timestamp=" + timestamp +
                ", author=" + author +
                ", auditType='" + auditType + '\'' +
                ", affectedObjects=" + affectedObjects +
                ", changedValues=" + changedValues +
                ", source='" + source + '\'' +
                ", system='" + system + '\'' +
                ", node='" + node + '\'' +
                ", method=" + method +
                ", extraAttributes=" + extraAttributes +
                '}';
    }

    public static Builder builder(AuditType type) {
        return new Builder(type);
    }

    /**
     * The builder class for {@link AuditEntity AuditEntity}
     */
    public static class Builder {

        private Long id;

        private Instant timestamp;

        private AuditAuthor author;

        private AuditType type;

        private List<AuditResource> affectedObjects = new ArrayList<>();

        private List<ChangedValue> changedValues = new ArrayList<>();

        private String source;

        private String system;

        private String node;

        private String method;

        private Set<AuditAttribute> extraAttributes = new HashSet<>();

        public Builder(AuditType type) {
            requireNonNull(type);

            this.type = type;
        }

        public Builder(AuditEntity entity) {
            this.id = entity.id;
            this.timestamp = entity.timestamp;
            this.author = entity.author;
            this.type = entity.auditType;
            this.affectedObjects = new ArrayList<>(entity.affectedObjects);
            this.changedValues = new ArrayList<>(entity.changedValues);
            this.source = entity.source;
            this.system = entity.system;
            this.node = entity.node;
            this.method = entity.method;
            this.extraAttributes = new HashSet<>(entity.extraAttributes);
        }

        public Builder id(Long id) {
            this.id = id;
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp.truncatedTo(MILLIS);
            return this;
        }

        public Builder author(AuditAuthor author) {
            this.author = author;
            return this;
        }

        public Builder type(AuditType type) {
            this.type = type;
            return this;
        }

        public Builder source(String source) {
            this.source = source;
            return this;
        }

        public Builder system(String system) {
            this.system = system;
            return this;
        }

        public Builder node(String node) {
            this.node = node;
            return this;
        }

        public Builder affectedObjects(@Nonnull List<AuditResource> affectedObjects) {
            this.affectedObjects = requireNonNull(affectedObjects);
            return this;
        }

        public Builder appendAffectedObjects(@Nonnull List<AuditResource> affectedObjects) {
            this.affectedObjects.addAll(requireNonNull(affectedObjects));
            return this;
        }

        public Builder affectedObject(AuditResource affectedObject) {
            affectedObjects.add(affectedObject);
            return this;
        }

        public Builder changedValues(@Nonnull List<ChangedValue> changedValues) {
            this.changedValues = requireNonNull(changedValues);
            return this;
        }

        public Builder appendChangedValues(@Nonnull Collection<ChangedValue> changedValues) {
            this.changedValues.addAll(requireNonNull(changedValues));
            return this;
        }

        public Builder changedValue(ChangedValue changedValue) {
            changedValues.add(changedValue);
            return this;
        }

        /**
         * Adds {@link ChangedValue} only if the value changed.
         *
         * @param changedValue New {@link ChangedValue} to be added.
         * @return {@link Builder} instance
         * @since 0.0.12
         */
        public Builder addChangedValueIfDifferent(ChangedValue changedValue) {
            if (!Objects.equals(changedValue.getFrom(), changedValue.getTo())) {
                changedValue(changedValue);
            }
            return this;
        }

        public Builder method(String method) {
            this.method = method;
            return this;
        }

        public Builder extraAttributes(@Nonnull Collection<AuditAttribute> extraAttributes) {
            this.extraAttributes = new HashSet<>(requireNonNull(extraAttributes));
            return this;
        }

        public Builder appendExtraAttributes(@Nonnull Collection<AuditAttribute> extraAttributes) {
            this.extraAttributes.addAll(requireNonNull(extraAttributes));
            return this;
        }

        public Builder extraAttribute(AuditAttribute extraAttribute) {
            extraAttributes.add(extraAttribute);
            return this;
        }

        public AuditEntity build() {
            if (timestamp == null) {
                timestamp = Instant.now().truncatedTo(MILLIS);
            }

            if (author == null) {
                author = AuditAuthor.SYSTEM_AUTHOR;
            }
            return new AuditEntity(this);
        }
    }
}
