package com.atlassian.audit.entity;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

/**
 * Represents a changed value caused by an audit event, used in {@link AuditEntity#getChangedValues()}
 */
public class ChangedValue {

    private final String key;

    private final String i18nKey;

    private final String from;

    private final String to;

    /**
     * @deprecated since 1.7.0 use {@link #fromI18nKeys(String)} and provide the changed value i18n key.
     */
    @Deprecated
    public ChangedValue(@Nonnull String key) {
        this(key, null, null);
    }

    /**
     * @deprecated since 1.7.0 use {@link #fromI18nKeys(String)} and provide the changed value i18n key.
     */
    @Deprecated
    public ChangedValue(@Nonnull String key, String to) {
        this(key, null, to);
    }

    /**
     * @deprecated since 1.7.0 use {@link #fromI18nKeys(String)} and provide the changed value i18n key.
     */
    @Deprecated
    public ChangedValue(@Nonnull String key, String from, String to) {
        this.key = requireNonNull(key);
        this.i18nKey = key;
        this.from = from;
        this.to = to;
    }

    /**
     * @since 1.7.0
     */
    public ChangedValue(Builder builder) {
        this.i18nKey = requireNonNull(builder.i18nKey);
        this.key = builder.keyTranslation;
        this.from = builder.from;
        this.to = builder.to;
    }

    public static Builder fromI18nKeys(@Nonnull String i18nKey) {
        return new Builder(i18nKey);
    }

    /**
     * @return the i18n key to identify the resource whose value changed.
     */
    @Nonnull
    public String getI18nKey() {
        return i18nKey;
    }

    /**
     * @return the translated resource identity whose value changed.
     */
    @Nonnull
    public String getKey() {
        return key;
    }

    /**
     * @return the old value.
     */
    @Nullable
    public String getFrom() {
        return from;
    }

    /**
     * @return the new value.
     */
    @Nullable
    public String getTo() {
        return to;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ChangedValue that = (ChangedValue) o;
        return Objects.equals(getKey(), that.getKey()) &&
                Objects.equals(i18nKey, that.i18nKey) &&
                Objects.equals(getFrom(), that.getFrom()) &&
                Objects.equals(getTo(), that.getTo());
    }

    @Override
    public String toString() {
        return "ChangedValue{" +
                "i18nKey='" + i18nKey + '\'' +
                ", key='" + key + '\'' +
                ", from='" + from + '\'' +
                ", to='" + to + '\'' +
                '}';
    }

    @Override
    public int hashCode() {
        return Objects.hash(i18nKey, key, from, to);
    }

    public static class Builder {
        private String i18nKey;
        private String keyTranslation;
        private String from;
        private String to;

        public Builder(@Nonnull String i18nKey) {
            this.i18nKey = requireNonNull(i18nKey);
        }

        public Builder withKeyTranslation(String keyTranslation) {
            this.keyTranslation = keyTranslation;
            return this;
        }

        public Builder to(String to) {
            this.to = to;
            return this;
        }

        public Builder from(String from) {
            this.from = from;
            return this;
        }

        public ChangedValue build() {
            return new ChangedValue(this);
        }
    }
}
