package com.atlassian.logging.log4j.layout.patterns;

import com.atlassian.logging.log4j.LogMessageUtil;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.pattern.ConverterKeys;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;
import org.apache.logging.log4j.core.pattern.PatternConverter;

/**
 * PatternParser that will suffix newline characters with a custom separator (default four spaces ). This is
 * to better highlight that log messages are being injected when logging user entered information. Using %nlm
 * instead of %m so that it doesn't conflict with the default message converter.
 *
 * @since v1.2
 */
@Plugin(name = "NewLineIndentingLogMessagePatternConverter", category = PatternConverter.CATEGORY)
@ConverterKeys({"nlm"})
public class NewLineIndentingLogMessagePatternConverter extends LogEventPatternConverter {
    // Line ending combos include UNIX and MAC = \n, WINDOWS = \r\n, MAC pre OSX = \r the next major OS will need to choose \n\r
    private final String lineIndent;

    public NewLineIndentingLogMessagePatternConverter(String[] options) {
        super("NewLineIndentingLogMessagePatternConverter", "NewLineIndentingLogMessagePatternConverter");
        this.lineIndent = extractLineIndentOption(options);
    }

    public static NewLineIndentingLogMessagePatternConverter newInstance(final String[] options) {
        return new NewLineIndentingLogMessagePatternConverter(options);
    }

    @Override
    public void format(LogEvent event, StringBuilder toAppendTo) {
        toAppendTo.append(LogMessageUtil.appendLineIndent(event.getMessage().getFormattedMessage(), lineIndent));
    }

    private String extractLineIndentOption(String[] options) {
        String lineIndent = "    ";
        int optionsCount = options.length;
        if (optionsCount == 1) {
            lineIndent = options[0];
        }
        return lineIndent;
    }
}
