package com.atlassian.logging.log4j.appender.fluentd;

import com.github.rholder.retry.Attempt;
import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.StopStrategy;
import com.github.rholder.retry.WaitStrategies;
import com.github.rholder.retry.WaitStrategy;

import java.io.Serializable;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * A TimerTask which sends off logging events to fluentd via http.
 * <p>
 * This is a single-thread task - therefore, a task will not start until the previous sendoff task has finished.
 */
public class FluentdLogQueueSendTask<T> extends TimerTask {
    /**
     * The max number of logs to send to fluentd in a single http request.  The calls will be batched by this number.
     * This is needed to not exceed the maximum http request size.
     */
    private static final int FLUENTD_BATCH_SIZE = 50;

    private final Function<T, Serializable> layout;
    private final LoggingEventQueue<T> loggingEventQueue;

    private final int maxRetryPeriodMs;
    private final int backoffMultiplier;
    private final int maxBackoffMinutes;

    private final Retryer<Void> sendoffRetryer;
    private final FluentdSender fluentdSender;

    public FluentdLogQueueSendTask(final Function<T, Serializable> layout,
                                   final LoggingEventQueue<T> loggingEventQueue,
                                   final FluentdSender fluentdSender,
                                   final int maxRetryPeriodMs,
                                   final int backoffMultiplier,
                                   final int maxBackoffMinutes) {
        this.layout = layout;
        this.loggingEventQueue = loggingEventQueue;
        this.fluentdSender = fluentdSender;
        this.maxRetryPeriodMs = maxRetryPeriodMs;
        this.backoffMultiplier = backoffMultiplier;
        this.maxBackoffMinutes = maxBackoffMinutes;

        sendoffRetryer = buildRetryer();
    }

    private Retryer<Void> buildRetryer() {
        /*
         * Stop retrying after 4 hours or when the log queue reaches configured size
         */
        final StopStrategy stopStrategy = new StopStrategy() {
            private final StopStrategy stopAfterDelay = StopStrategies.stopAfterDelay(maxRetryPeriodMs, TimeUnit.MILLISECONDS);

            @Override
            public boolean shouldStop(Attempt failedAttempt) {
                if (stopAfterDelay.shouldStop(failedAttempt)) {
                    return true;
                }
                // If we've reached the max queue size, just drop these logs so we can try the next batch instead.
                return loggingEventQueue.isFull();
            }
        };

        /*
         * Exponential + random wait between retries.
         */
        final WaitStrategy waitStrategy = new WaitStrategy() {
            public static final double RANDOM_RANGE_PERCENT = 1.2; // Up do 20% higher

            private final Random RANDOM = new Random();

            private WaitStrategy exponentialWait = WaitStrategies.exponentialWait(backoffMultiplier, maxBackoffMinutes, TimeUnit.MINUTES);

            @Override
            public long computeSleepTime(Attempt failedAttempt) {
                final long minimum = exponentialWait.computeSleepTime(failedAttempt);
                final long maximum = (long) (minimum * RANDOM_RANGE_PERCENT);
                return minimum + Math.abs(RANDOM.nextLong()) % (maximum - minimum);
            }
        };

        return RetryerBuilder.<Void>newBuilder()
                .retryIfExceptionOfType(FluentdRetryableException.class)
                .retryIfRuntimeException()
                .withWaitStrategy(waitStrategy)
                .withStopStrategy(stopStrategy)
                .build();
    }

    public void run() {
        final List<T> eventsToSend = loggingEventQueue.retrieveLoggingEvents(FLUENTD_BATCH_SIZE);

        if (eventsToSend.isEmpty()) {
            return;
        }
        try {
            final String payload = buildPayload(eventsToSend);

            sendoffRetryer.call(() -> {
                try {
                    fluentdSender.send(payload);
                } catch (Exception e) {
                    System.err.println("Error in attempt to send logs to FluentD");
                    e.printStackTrace(System.err);
                    throw e;
                }
                return null;
            });
        } catch (RetryException e) {
            // All attempts failed.  These logs are lost forever!
            System.err.println("FluentD logging failed - " + eventsToSend.size() + " logs lost");
            e.printStackTrace(System.err);
        } catch (ExecutionException e) {
            // Any exception the callable throws besides RuntimeException and RetryableException
            // .. This should not be even possible
            System.err.println("FluentD logging failed for unknown reason");
            e.printStackTrace(System.err);
        }
    }

    /**
     * Attemtps to send a last batch of messages.
     * You can call this method after canceling the task.
     */
    public void clean() {
        // Flush queue, if there many are pending messages they may be lost;
        final List<T> loggingEvents = loggingEventQueue.retrieveLoggingEvents(FLUENTD_BATCH_SIZE);
        if (loggingEvents.isEmpty()) {
            return;
        }
        if (loggingEventQueue.getSize() > 0) {
            System.err.println("There are pending log messages that will be lost");
        }
        try {
            final String payload = buildPayload(loggingEvents);
            fluentdSender.send(payload);
        } catch (Exception e) {
            System.err.println("Error in attempt to send logs to FluentD");
        }
    }

    private String buildPayload(final List<T> loggingEvents) {
        final String payload = loggingEvents.stream()
                .map(this::formatOptionally)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.joining(","));

        return "[" + payload + "]";
    }

    private Optional<String> formatOptionally(final T loggingEvent) {
        try {
            final Serializable formattedEvent = layout.apply(loggingEvent);
            return Optional.of(formattedEvent.toString());
        } catch (final Exception e) {
            // We don't want a single failure to format a log event stopping everything else in the batch
            System.err.println("Could not format event for logger:" + loggingEvent);
            e.printStackTrace(System.err);
            return Optional.empty();
        }
    }

}
