package com.atlassian.logging.log4j;

import org.apache.logging.log4j.core.impl.ExtendedStackTraceElement;
import org.apache.logging.log4j.core.impl.ThrowableProxy;
import org.apache.logging.log4j.core.util.Throwables;
import org.apache.logging.log4j.util.Strings;

import javax.annotation.Nonnull;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@SuppressWarnings("UnusedDeclaration")
public class StackTraceInfo {
    private final String lineIndent;
    private final String[] stackTrace;
    private final boolean stackTracePackagingExamined;

    /**
     * Constructs a StackTraceInfo object that gives back a stack trace as a series of lines with packaging information examined
     *
     * @param throwable  the throwable containing the stack trace
     * @param lineIndent the line indent to use
     */
    public StackTraceInfo(final @Nonnull Throwable throwable, final @Nonnull String lineIndent) {
        this(throwable, lineIndent, true);
    }

    /**
     * Constructs a StackTraceInfo object that gives back a stack trace as a series of lines
     *
     * @param throwable                   the throwable containing the stack trace
     * @param lineIndent                  the line indent to use
     * @param stackTracePackagingExamined whether the packaging information of stack trace elements should be examined
     */
    public StackTraceInfo(final @Nonnull Throwable throwable, final @Nonnull String lineIndent, final boolean stackTracePackagingExamined) {
        this.lineIndent = lineIndent;
        this.stackTracePackagingExamined = stackTracePackagingExamined;
        this.stackTrace = buildThrowableStrRep(throwable);
    }

    /**
     * Allows you to get a string representation from throwable.  This is kinda the equivalent you might get from throwable.printStackTrace()
     *
     * @param throwable the throwable
     * @return a String representation of the Throwable
     */
    public static String asString(final @Nonnull Throwable throwable) {
        return new StackTraceInfo(throwable, CommonConstants.DEFAULT_NEW_LINE_PREFIX, true).asString();
    }

    /**
     * Allows you to get a string representation from throwable.  This is kinda the equivalent you might get from
     * throwable.printStackTrace()
     *
     * @param throwable the throwable
     * @return a List representation of the Throwable
     */
    public static List<String> asLines(final @Nonnull Throwable throwable) {
        return new StackTraceInfo(throwable, CommonConstants.DEFAULT_NEW_LINE_PREFIX, true).asLines();
    }


    /**
     * Modelled on the original log4j usage it returns arrays of stack trace lines.
     *
     * @return an array of stack trace lines
     */
    public String[] getThrowableStrRep() {
        return stackTrace;
    }

    /**
     * Allows you to get a string representation from throwable.  This is kinda the equivalent you might get from throwable.printStackTrace()
     *
     * @return a String representation of the stack trace
     */
    public String asString() {
        return NewLineSupport.join(stackTrace);
    }

    /**
     * This returns the same as {@link #getThrowableStrRep()} except as a list
     *
     * @return a list of stack trace lines
     */
    public List<String> asLines() {
        return Arrays.asList(stackTrace);
    }

    private String[] buildThrowableStrRep(final @Nonnull Throwable throwable) {
        try {
            if (stackTracePackagingExamined) {
                return buildPackagingRepresentation(throwable);
            } else {
                return buildNoPackagingRepresentation(throwable);
            }
        } catch (RuntimeException rte) {
            //
            // should we, when trying to show extended stack trace information, end up throwing an exception ourselves
            // then let's use Java 7 suppressed functionality to squirrel that away and fall back to to the old way from log4j
            //
            // Something is seriously wrong with the code if we ever get here but at least we don't barf when handling other peoples barf.
            //
            throwable.addSuppressed(rte);
            return buildNoPackagingRepresentation(throwable);
        }
    }

    /**
     * Builds an old school stack trace representation like log4j used to do
     *
     * @param throwable the throwable in play
     * @return the stack trace as lines
     */
    private String[] buildNoPackagingRepresentation(final Throwable throwable) {
        return Throwables.toStringList(throwable).toArray(Strings.EMPTY_ARRAY);
    }


    /**
     * Builds a stack trace representation that includes code packaging information
     *
     * @param throwable the throwable in play
     * @return the stack trace as lines, with packaging information in there
     */
    private String[] buildPackagingRepresentation(Throwable throwable) {
        ThrowableProxy throwableProxy = new ThrowableProxy(throwable);

        List<String> lines = new ArrayList<>();
        final String message = throwable.toString();

        lines.add(LogMessageUtil.appendLineIndent(message, lineIndent));
        StackTraceElement[] stackTraceElements = throwable.getStackTrace();
        addStackTraceLines(lines, "\t", throwableProxy.getExtendedStackTrace(), stackTraceElements.length - 1);

        addSuppressedThrowableInformation(lines, "", throwableProxy, stackTraceElements);

        Throwable cause = throwable.getCause();
        if (cause != null) {
            addCausedByThrowableInformation(lines, "", cause, throwableProxy.getCauseProxy(), stackTraceElements);
        }
        return lines.toArray(new String[lines.size()]);
    }

    private void addCausedByThrowableInformation(final List<String> lines, final String tabs, final Throwable cause, final ThrowableProxy causeProxy, final StackTraceElement[] parentStackTrace) {
        final String message = cause.toString();

        StackTraceElement[] stackTraceElements = cause.getStackTrace();
        ExtendedStackTraceElement[] extendedStackTrace = causeProxy.getExtendedStackTrace();

        final int endUniqueFrames = Math.min(getEndOfUniqueFrames(parentStackTrace, stackTraceElements), extendedStackTrace.length);

        lines.add(tabs + "Caused by: " + LogMessageUtil.appendLineIndent(message, lineIndent));
        addStackTraceLines(lines, tabs + "\t", extendedStackTrace, endUniqueFrames);

        int more = stackTraceElements.length - 1 - endUniqueFrames;
        if (more > 0) {
            lines.add(tabs + "\t... " + more + " more");
        }

        addSuppressedThrowableInformation(lines, tabs + "\t", causeProxy, stackTraceElements);


        if (cause.getCause() != null) {
            addCausedByThrowableInformation(lines, tabs, cause.getCause(), causeProxy.getCauseProxy(), stackTraceElements);
        }
    }

    @SuppressWarnings("ThrowableResultOfMethodCallIgnored")
    private void addSuppressedThrowableInformation(final List<String> lines, final String tabs, final ThrowableProxy throwableProxy, final StackTraceElement[] parentStackTrace) {
        ThrowableProxy[] suppressedTs = throwableProxy.getSuppressedProxies();
        for (ThrowableProxy suppressed : suppressedTs) {
            final String message = suppressed.getThrowable().toString();

            StackTraceElement[] stackTraceElements = suppressed.getStackTrace();
            ExtendedStackTraceElement[] extendedStackTrace = suppressed.getExtendedStackTrace();

            final int endUniqueFrames = Math.min(getEndOfUniqueFrames(parentStackTrace, stackTraceElements), extendedStackTrace.length);

            lines.add(tabs + "Suppressed: " + LogMessageUtil.appendLineIndent(message, lineIndent));
            addStackTraceLines(lines, tabs + "\t", extendedStackTrace, endUniqueFrames);

            int more = stackTraceElements.length - 1 - endUniqueFrames;
            if (more > 0) {
                lines.add(tabs + "\t... " + more + " more");
            }

            if (suppressed.getCauseProxy() != null) {
                addCausedByThrowableInformation(lines, tabs, suppressed.getThrowable().getCause(), suppressed.getCauseProxy(), stackTraceElements);
            }
        }
    }

    private String fmt(ExtendedStackTraceElement element) {
        // we don't want the ~ exact thing of ExtendedStackTraceElement.toString().  Its pointless.  Plus we want collapsing of
        // stack trace names
        return String.valueOf(element.getStackTraceElement()) + " " + '[' + element.getLocation() + ':' + element.getVersion() + ']';
    }

    private void addStackTraceLines(final List<String> lines, final String tabs, final ExtendedStackTraceElement[] extendedStackTrace, final int endPos) {
        for (int i = 0; i <= endPos; i++) {
            lines.add(tabs + "at " + fmt(extendedStackTrace[i]));
        }
    }

    private int getEndOfUniqueFrames(final StackTraceElement[] parentElements, final StackTraceElement[] causeElements) {
        int i = parentElements.length - 1;
        int causeStart = causeElements.length - 1;
        while (causeStart >= 0 && i >= 0 && causeElements[causeStart].equals(parentElements[i])) {
            --causeStart;
            --i;
        }
        return causeStart;
    }

    public void printStackTrace(final PrintWriter pw) {
        for (String line : getThrowableStrRep()) {
            pw.println(line);
        }
    }

    public void printStackTrace(final PrintStream ps) {
        for (String line : getThrowableStrRep()) {
            ps.println(line);
        }
    }
}
