package com.atlassian.logging.log4j;

import org.apache.commons.lang3.StringUtils;

/**
 * Modelled on the log back converters, but with a clean room implementation (honest), this will take a FQN like
 * "com.atlassian.logging.log4j.FqNameCollapser" and collapse it to a target length from the right most side or it uses
 * a segment strategy to collapse it to a set number of segments and first character shortening after that.
 * <br>
 * Its length strategy rules are :
 * <pre>
 *   - never collapse the right most name - ie the class name
 *   - aim to output at least the specified precision of package names
 *   - never partially render a package name
 *   - never exceed the precision on output
 *   - and hence collapse the package name if it will only partially fit
 *
 *   com.atlassian.logging.log4j.FqNameCollapser (10) becomes c.a.l.log4j.FqNameCollapser
 *
 * </pre>
 * Its segment strategy rules are :
 * <pre>
 *   - never collapse the right most name - ie the class name
 *   - aim to output at least the specified precision of package names
 *   - never partially render a package name
 *   - never exceed the precision on output
 *
 *   com.atlassian.logging.log4j.FqNameCollapser (2) becomes c.a.logging.log4j.FqNameCollapser
 *
 * </pre>
 */
@SuppressWarnings("UnusedDeclaration")
public class FqNameCollapser {
    public static final int NO_COLLAPSING_PRECISION = -1;

    /**
     * What strategy to use
     */
    public enum Strategy {
        PACKAGE_LENGTH, PACKAGE_SEGMENTS
    }

    /**
     * Use this collapser if you want NO collapsing to occur
     */
    public static FqNameCollapser NO_COLLAPSING = new FqNameCollapser(NO_COLLAPSING_PRECISION, Strategy.PACKAGE_LENGTH);

    private final int fqNameCollapsePrecision;
    private final Strategy strategy;

    /**
     * <pre>
     *     less then 0  - means never shorten it
     *     0    - means shorten it to just the class name
     *     greater then 0  - try to shorten it but follow the rules
     * </pre>
     *
     * @param fqNameCollapsePrecision the precision in play
     * @param strategy                what collapsing strategy to use
     */
    public FqNameCollapser(final int fqNameCollapsePrecision, final Strategy strategy) {
        this.fqNameCollapsePrecision = fqNameCollapsePrecision;
        this.strategy = strategy;
    }

    public String collapse(final String fqName) {
        if (fqNameCollapsePrecision < 0) {
            return fqName;
        }
        if (StringUtils.isBlank(fqName)) {
            return fqName;
        }
        String[] packages = StringUtils.splitPreserveAllTokens(fqName, ".");
        String rightMostClassName = packages[packages.length - 1];
        if (packages.length < 2) {
            return fqName;
        }
        if (fqNameCollapsePrecision == 0) {
            return rightMostClassName;
        }
        if (strategy == Strategy.PACKAGE_SEGMENTS) {
            return segmentStrategy(fqName.length(), packages, rightMostClassName);
        } else {
            return lengthStrategy(fqName.length(), packages, rightMostClassName);
        }
    }

    private String segmentStrategy(final int length, final String[] packages, final String rightMostClassName) {
        StringBuilder sb = new StringBuilder(length);
        sb.append(".").append(rightMostClassName);
        int segmentCount = 1;
        for (int i = packages.length - 2; i >= 0; i--) {
            String packageName = StringUtils.defaultString(packages[i]);
            if (packageName.length() > 0) {
                if (segmentCount <= fqNameCollapsePrecision) {
                    sb.insert(0, packageName);
                } else {
                    sb.insert(0, firstCharOf(packageName));
                }
                segmentCount++;
            }
            if (i > 0) {
                sb.insert(0, ".");
            }
        }
        return sb.toString();
    }

    private String lengthStrategy(final int length, final String[] packages, final String rightMostClassName) {
        StringBuilder sb = new StringBuilder(length);
        int spaceLeft = fqNameCollapsePrecision;
        sb.append(".").append(rightMostClassName);
        for (int i = packages.length - 2; i >= 0; i--) {
            String packageName = StringUtils.defaultString(packages[i]);
            if (packageName.length() > 0) {
                if (spaceLeft <= fqNameCollapsePrecision) {
                    // will the package name fit
                    if (packageName.length() <= spaceLeft) {
                        sb.insert(0, packageName);
                    } else {
                        sb.insert(0, firstCharOf(packageName));
                    }
                    spaceLeft -= packageName.length();
                } else {
                    sb.insert(0, firstCharOf(packageName));
                }
            }
            if (i > 0) {
                sb.insert(0, ".");
            }
        }
        return sb.toString();
    }

    private String firstCharOf(final String packageName) {
        return packageName.length() > 1 ? packageName.substring(0, 1) : packageName;
    }
}
