package com.atlassian.logging.log4j.layout.patterns;

import static com.atlassian.logging.log4j.NewLineSupport.NL;

public class StackTraceFilteringPatternConverterOptions {

    private static final StackTraceFilteringPatternConverterOptions DEFAULT = new StackTraceFilteringPatternConverterOptions();


    /**
     * How many lines to always show
     */
    public static final String MINIMUM_LINES = "minimumLines";
    private int minimumLines = 6;

    /**
     * Whether to show a summary of eluded lines
     */
    public static final String SHOW_ELUDED_SUMMARY = "showEludedSummary";
    private boolean showEludedSummary = false;

    /**
     * Holds the list of filtered frames.
     */
    public static final String FILTERED_FRAMES = "filteredFrames";
    private String filteredFrames;

    /**
     * Holds the list of everyThingAfter filtered frames.
     */
    public static final String FILTER_EVERYTHING_AFTER_FRAMES = "filterEverythingAfterFrames";
    private String filterEverythingAfterFrames;

    /**
     * The message to output if we filter everything after that point
     */
    public static final String FILTER_EVERYTHING_AFTER_MESSAGE = "filterEverythingAfterMessage";
    private String filterEverythingAfterMessage = "\t\t(The rest of the stack trace has been filtered ...)";

    /**
     * Holds the list of frames that will generate a marker
     */
    public static final String MARKER_AT_FRAMES = "markerAtFrames";
    private String markerAtFrames;

    /**
     * The message to mark specific lines at
     */
    public static final String MARKER_AT_MESSAGE = "markerAtMessage";
    private String markerAtMessage = NL;

    /**
     * The replacement string if a filter is applied to a stack trace line.  Default is "... "
     */
    public static final String FILTER_REPLACEMENT_TOKEN = "filterReplacementToken";
    private String filterReplacementToken = "... ";

    /**
     * This is the master switch on whether filtering will be applied or not.
     */
    public static final String FILTERING_APPLIED = "filteringApplied";
    private boolean filteringApplied = false;

    /**
     * This controls whether Debug level messages are filtered or not.  By default they are NOT.
     */
    public static final String FILTERING_APPLIED_TO_DEBUG_LEVEL = "filteringAppliedToDebugLevel";
    private boolean filteringAppliedToDebugLevel = false;

    /**
     * This is the master switch on whether stack trace package examination will be performed or not.  This also allows it to be switched on or
     * off at run time
     */
    public static final String STACKTRACE_PACKAGING_EXAMINED = "stackTracePackagingExamined";
    private boolean stackTracePackagingExamined = true;

    private StackTraceFilteringPatternConverterOptions() {
    }

    public StackTraceFilteringPatternConverterOptions(int minimumLines,
                                                      boolean showEludedSummary,
                                                      String filteredFrames,
                                                      String filterEverythingAfterFrames,
                                                      String filterEverythingAfterMessage,
                                                      String markerAtFrames,
                                                      String markerAtMessage,
                                                      String filterReplacementToken,
                                                      boolean filteringApplied,
                                                      boolean filteringAppliedToDebugLevel,
                                                      boolean stackTracePackagingExamined) {
        this.minimumLines = minimumLines;
        this.showEludedSummary = showEludedSummary;
        this.filteredFrames = filteredFrames;
        this.filterEverythingAfterFrames = filterEverythingAfterFrames;
        this.filterEverythingAfterMessage = filterEverythingAfterMessage;
        this.markerAtFrames = markerAtFrames;
        this.markerAtMessage = markerAtMessage;
        this.filterReplacementToken = filterReplacementToken;
        this.filteringApplied = filteringApplied;
        this.filteringAppliedToDebugLevel = filteringAppliedToDebugLevel;
        this.stackTracePackagingExamined = stackTracePackagingExamined;
    }

    public static StackTraceFilteringPatternConverterOptions newInstance(String[] options) {
        if (options == null || options.length == 0) {
            return DEFAULT;
        }

        int minimumLines = DEFAULT.getMinimumLines();
        boolean showEludedSummary = DEFAULT.isShowEludedSummary();
        String filteredFrames = DEFAULT.getFilteredFrames();
        String filterEveryThingAfterFrames = DEFAULT.getFilterEveryThingAfterFrames();
        String filterEveryThingAfterMessage = DEFAULT.getFilterEverythingAfterMessage();
        String markerAtFrames = DEFAULT.getMarkerAtFrames();
        String markerAtMessage = DEFAULT.getMarkerAtMessage();
        String filterReplacementToken = DEFAULT.getFilterReplacementToken();
        boolean filteringApplied = DEFAULT.isFilteringApplied();
        boolean filteringAppliedToDebugLevel = DEFAULT.isFilteringAppliedToDebugLevel();
        boolean stackTracePackagingExamined = DEFAULT.isStackTracePackagingExamined();


        for (final String rawOption : options) {
            if (rawOption != null) {
                final String option = rawOption.trim();
                if (option.isEmpty()) {
                    // continue;
                } else if (isOption(MINIMUM_LINES, option)) {
                    minimumLines = Integer.parseInt(getOptionValue(MINIMUM_LINES, option));
                } else if (isOption(SHOW_ELUDED_SUMMARY, option)) {
                    showEludedSummary = Boolean.parseBoolean(getOptionValue(SHOW_ELUDED_SUMMARY, option));
                } else if (isOption(FILTERED_FRAMES, option)) {
                    filteredFrames = getOptionValue(FILTERED_FRAMES, option);
                } else if (isOption(FILTER_EVERYTHING_AFTER_FRAMES, option)) {
                    filterEveryThingAfterFrames = getOptionValue(FILTER_EVERYTHING_AFTER_FRAMES, option);
                } else if (isOption(FILTER_EVERYTHING_AFTER_MESSAGE, option)) {
                    filterEveryThingAfterMessage = getOptionValue(FILTER_EVERYTHING_AFTER_MESSAGE, option);
                } else if (isOption(MARKER_AT_FRAMES, option)) {
                    markerAtFrames = getOptionValue(MARKER_AT_FRAMES, option);
                } else if (isOption(MARKER_AT_MESSAGE, option)) {
                    markerAtMessage = getOptionValue(MARKER_AT_MESSAGE, option);
                } else if (isOption(FILTER_REPLACEMENT_TOKEN, option)) {
                    filterReplacementToken = getOptionValue(FILTER_REPLACEMENT_TOKEN, option);
                } else if (isOption(FILTERING_APPLIED, option)) {
                    filteringApplied = Boolean.parseBoolean(getOptionValue(FILTERING_APPLIED, option));
                } else if (isOption(FILTERING_APPLIED_TO_DEBUG_LEVEL, option)) {
                    filteringAppliedToDebugLevel = Boolean.parseBoolean(getOptionValue(FILTERING_APPLIED_TO_DEBUG_LEVEL, option));
                } else if (isOption(STACKTRACE_PACKAGING_EXAMINED, option)) {
                    stackTracePackagingExamined = Boolean.parseBoolean(getOptionValue(STACKTRACE_PACKAGING_EXAMINED, option));
                }
            }
        }
        return new StackTraceFilteringPatternConverterOptions(minimumLines,
                showEludedSummary,
                filteredFrames,
                filterEveryThingAfterFrames,
                filterEveryThingAfterMessage,
                markerAtFrames,
                markerAtMessage,
                filterReplacementToken,
                filteringApplied,
                filteringAppliedToDebugLevel,
                stackTracePackagingExamined);
    }

    private static boolean isOption(String option, String toMatch) {
        return toMatch.startsWith(option + "(") && toMatch.endsWith(")");
    }

    private static String getOptionValue(String option, String toExtract) {
        return toExtract.substring((option + "(").length(), toExtract.length() - 1);
    }

    public int getMinimumLines() {
        return minimumLines;
    }

    public boolean isShowEludedSummary() {
        return showEludedSummary;
    }

    public String getFilteredFrames() {
        return filteredFrames;
    }

    public String getFilterEveryThingAfterFrames() {
        return filterEverythingAfterFrames;
    }

    public String getFilterEverythingAfterMessage() {
        return filterEverythingAfterMessage;
    }

    public String getMarkerAtFrames() {
        return markerAtFrames;
    }

    public String getMarkerAtMessage() {
        return markerAtMessage;
    }

    public String getFilterReplacementToken() {
        return filterReplacementToken;
    }

    public boolean isFilteringApplied() {
        return filteringApplied;
    }

    public boolean isFilteringAppliedToDebugLevel() {
        return filteringAppliedToDebugLevel;
    }

    public boolean isStackTracePackagingExamined() {
        return stackTracePackagingExamined;
    }
}
