package com.atlassian.logging.log4j.layout.patterns;

import com.atlassian.logging.log4j.FqNameCollapser;
import com.google.common.collect.ImmutableMap;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.pattern.ConverterKeys;
import org.apache.logging.log4j.core.pattern.LogEventPatternConverter;
import org.apache.logging.log4j.core.pattern.PatternConverter;

import java.util.Map;

/**
 * This pattern parser can interpret %q{n}{strategy} and shorten the category more smartly than log4j does.
 * 'n' can be any integer. Not collapsing by default.
 * 'strategy' can be 'length' or 'segments' (and 'l' or 's'). The default is segments.
 */
@Plugin(name = "CategoryCollapsingPatternConverter", category = PatternConverter.CATEGORY)
@ConverterKeys({"q", "Q"})
public class CategoryCollapsingPatternConverter extends LogEventPatternConverter {

    Map<String, FqNameCollapser.Strategy> strategyOptionMap = ImmutableMap.of(
            "length", FqNameCollapser.Strategy.PACKAGE_LENGTH,
            "segments", FqNameCollapser.Strategy.PACKAGE_SEGMENTS,
            "l", FqNameCollapser.Strategy.PACKAGE_LENGTH,
            "s", FqNameCollapser.Strategy.PACKAGE_SEGMENTS
    );
    private final FqNameCollapser fqNameCollapser;

    public CategoryCollapsingPatternConverter(String[] options) {
        super("CategoryCollapsingPatternConverter", "CategoryCollapsingPatternConverter");
        fqNameCollapser = new FqNameCollapser(extractPrecisionOption(options), extractStrategyOption(options));
    }

    public static CategoryCollapsingPatternConverter newInstance(final String[] options) {
        return new CategoryCollapsingPatternConverter(options);
    }

    @Override
    public void format(LogEvent event, StringBuilder toAppendTo) {
        toAppendTo.append(fqNameCollapser.collapse(event.getLoggerName()));
    }

    private int extractPrecisionOption(String[] options) {
        int collapsePrecision = -1;
        int optionsCount = options.length;
        if (optionsCount >= 1) {
            try {
                collapsePrecision = Integer.parseInt(options[0]);
            } catch (NumberFormatException exception) {
                return collapsePrecision;
            }
        }
        return collapsePrecision;
    }

    private FqNameCollapser.Strategy extractStrategyOption(String[] options) {
        FqNameCollapser.Strategy strategy = FqNameCollapser.Strategy.PACKAGE_SEGMENTS;
        int optionsCount = options.length;
        if (optionsCount >= 2) {
            String strategyString = options[1];
            strategy = strategyOptionMap.getOrDefault(strategyString, FqNameCollapser.Strategy.PACKAGE_SEGMENTS);
        }
        return strategy;
    }
}
