package com.atlassian.logging.log4j.layout.json;

import com.atlassian.json.marshal.Jsonable;
import com.atlassian.logging.log4j.StackTraceCompressor;
import com.atlassian.logging.log4j.layout.AtlassianJsonLayout;
import com.atlassian.logging.log4j.util.CleanLogging;
import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import org.apache.logging.log4j.util.ReadOnlyStringMap;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonGenerator;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;

/**
 * Produces json logs based on this specification:
 * https://pug.jira-dev.com/wiki/display/CP/RFC+-+Consistent+JSON+logging+format+for+application+logs
 * <br>
 * Can eliminate unnecessary stacktrace lines
 * <br>
 * Settings:
 * <br>
 * * dataProvider - path.to.DataProvider - defaults to JsonDataProvider.  Must extend JsonDataProviderInterface
 * This class is used to inject data into the json structure.  Please see {@link DefaultJsonDataProvider} for
 * default implementation.
 * <br>
 * * FilterFrames - List of class/package names to be filtered out from stacktrace.
 * Please see {@link StackTraceCompressor.Builder#filteredFrames(String)} for the syntax.
 * <p>
 * NOTE: does not seem conform https://pug.jira-dev.com/wiki/display/CC/RFC+-+Consistent+JSON+logging+format+for+application+logs
 * (for example, produces serviceId field that MUST NOT be part of message)
 */
public class JsonLayoutHelper {
    @FunctionalInterface
    private interface CheckedValueConsumer<T> {
        void accept(String field, T value) throws Exception;
    }

    @Override
    public String toString() {
        return "JsonLayoutHelper{" +
                "dataProvider=" + dataProvider.getClass().toString() +
                ", suppressedFields=" + suppressedFields +
                ", includeLocation=" + includeLocation +
                ", filteringApplied=" + filteringApplied +
                ", minimumLines=" + minimumLines +
                ", showEludedSummary=" + showEludedSummary +
                ", filteredFrames='" + filteredFrames + '\'' +
                ", additionalFields=" + additionalFields +
                '}';
    }

    @FunctionalInterface
    private interface CheckedValueProducer<T> {
        T get() throws Exception;
    }

    private static final String TIME_ZONE = "UTC";
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss,S'Z'");

    static {
        DATE_FORMAT.setTimeZone(TimeZone.getTimeZone(TIME_ZONE));
    }

    private static final Joiner STACK_TRACE_JOINER = Joiner.on("\n");

    /**
     * Application should set this in the log4j configuration.  Defaults to JsonDataProvider. This class is used to inject
     * data into the json structure.  Please see JsonDataProvider for default implementation.
     */
    private JsonDataProvider dataProvider;

    private final JsonFactory jsonFactory = new JsonFactory();

    private JsonStaticData staticData;

    private StackTraceCompressor stackTraceCompressor;

    private Set<String> suppressedFields = Collections.emptySet();

    private CleanLogging cleanLogging = new CleanLogging();

    // See https://jdog.jira-dev.com/browse/JPERF-1286
    private boolean includeLocation = false;

    /**
     * This is the master switch on whether filtering will be applied or not. This also allows it to be switched on or
     * off at run time
     */
    private boolean filteringApplied = true;

    /**
     * How many lines to always show
     */
    private int minimumLines = 6;

    /**
     * Whether to show a summary of eluded lines
     */
    private boolean showEludedSummary = false;

    /**
     * Holds the list of filtered frames.
     */
    private String filteredFrames;

    /**
     * Hold additional tags that will be added to events
     */
    private Map<String, String> additionalFields = Maps.newHashMap();

    private static final class JSON_KEYS {
        private static final String TIMESTAMP = "timestamp";
        private static final String LEVEL = "level";
        private static final String SERVICE_ID = "serviceId";
        private static final String PRODUCT = "product";
        private static final String HOST_NAME = "hostname";
        private static final String ENVIRONMENT = "env";
        private static final String ENV_SUFFIX = "env_suffix";
        private static final String PROCESS_ID = "pid";
        private static final String THREAD = "thread";
        private static final String LOGGER = "logger";
        private static final String MESSAGE = "message";

        private static final String ERROR = "err";
        private static final String ERROR_CLASS = "class";
        private static final String ERROR_MESSAGE = "msg";
        private static final String STACK_TRACE = "stack";

        private static final String CONTEXT = "ctx";
        private static final String REQUEST_ID = "requestId";
        private static final String SESSION_ID = "sessionId";
        private static final String USER_KEY = "userKey";
        private static final String UNICORN = "unicorn";
        private static final String DATA_CENTER = "dc";
        private static final String RACK = "rack";

        private static final String LOCATION = "location";
        private static final String CLASS = "class";
        private static final String METHOD = "method";
        private static final String LINE_NUMBER = "line";

        private static final String EXTRA = "ext";
    }

    public String format(final org.apache.logging.log4j.core.LogEvent logEvent) {
        return format(wrap(logEvent));
    }

    public String format(final LogEvent event) {
        StringWriter stringWriter = new StringWriter();

        Optional<String> splitEnv = cleanLogging.getEnvironmentSuffixForSplit(event.getLoggerName());
        if (splitEnv.isPresent()) {
            // just send to the alternate ENV
            formatWithEnv(event, splitEnv, stringWriter);
        } else {
            // send to primary ENV and maybe copy it to another
            formatWithEnv(event, Optional.empty(), stringWriter);
            Optional<String> copyEnv = cleanLogging.getEnvironmentSuffixForCopy(event.getLoggerName());
            if (copyEnv.isPresent()) {
                stringWriter.append(",");
                formatWithEnv(event, copyEnv, stringWriter);
            }
        }

        return stringWriter.toString();
    }

    private LogEvent wrap(org.apache.logging.log4j.core.LogEvent event) {
        return new LogEvent() {
            @SuppressWarnings("unchecked")
            @Override
            public org.apache.logging.log4j.core.LogEvent getNativeLogEvent() {
                return event;
            }

            @Override
            public String getLoggerName() {
                return event.getLoggerName();
            }

            @Override
            public String getMessage() {
                return event.getMessage() == null ? "" : event.getMessage().getFormattedMessage();
            }

            @Override
            public long getTimestamp() {
                return event.getTimeMillis();
            }

            @Override
            public String getLevel() {
                return event.getLevel().toString();
            }

            @Override
            public String getThreadName() {
                return event.getThreadName();
            }

            @Override
            public Throwable getThrown() {
                return event.getThrown();
            }

            @Override
            public StackTraceElement[] getStackTraceElements() {
                return event.getThrown().getStackTrace();
            }

            @Override
            public ReadOnlyStringMap getThreadContextMap() {
                return event.getContextData();
            }

            @Override
            public LocationInfo getLocationInformation() {
                final StackTraceElement info = event.getSource();
                return new LocationInfo() {
                    @Override
                    public String getClassName() {
                        return info.getClassName();
                    }

                    @Override
                    public String getMethodName() {
                        return info.getMethodName();
                    }

                    @Override
                    public String getLineNumber() {
                        return String.valueOf(info.getLineNumber());
                    }
                };
            }
        };
    }

    private void formatWithEnv(
            LogEvent event,
            Optional<String> suffix,
            StringWriter stringWriter) {
        try {
            JsonGenerator g = jsonFactory.createJsonGenerator(stringWriter);
            g.writeStartObject();

            writeFields(event, suffix, g);

            g.writeEndObject();
            g.close();
        } catch (IOException e) {
            throw new RuntimeException("JsonLayout - Failed to format", e);
        }

        stringWriter.append("\n");
    }

    /**
     * Suppresses output of specified fields
     *
     * @param suppressedFields field names separated by commas
     */
    public void setSuppressedFields(String suppressedFields) {
        this.suppressedFields = new HashSet<>(stream(suppressedFields.split(","))
                .map(String::trim)
                .collect(toList()));
    }

    public void setIncludeLocation(boolean includeLocation) {
        this.includeLocation = includeLocation;
    }

    // JDEV-37311: This setter is intentionally a different name to "setDataProvider()" in order to fix a bug.
    // See https://jdog.jira-dev.com/browse/JDEV-37311 for details.
    public void setDataProvider(final JsonDataProvider dataProvider) {
        this.dataProvider = dataProvider;
    }

    /**
     * Controls whether filtering will be applied to via this pattern
     *
     * @param filteringApplied the boolean flag
     */
    public void setFilteringApplied(boolean filteringApplied) {
        this.filteringApplied = filteringApplied;
    }

    /**
     * Sets how many first N lines of the stack trace are always shown
     *
     * @param minimumLines how many lines to always show
     */
    public void setMinimumLines(int minimumLines) {
        this.minimumLines = minimumLines;
    }

    /**
     * Allows the eluded line to be shown in horizontal summary form
     *
     * @param showEludedSummary true if they are to be shown
     */
    public void setShowEludedSummary(boolean showEludedSummary) {
        this.showEludedSummary = showEludedSummary;
    }

    /**
     * Any stack frame starting with <code>"at "</code> + <code>filter</code> will not be written to the log.
     * <br>
     * You can specify multiples frames separated by commas eg "org.apache.tomcat, org.hibernate"
     *
     * @param filteredFrames class name(s) or package name(s) to be filtered
     */
    public void setFilteredFrames(String filteredFrames) {
        this.filteredFrames = filteredFrames;
    }

    /**
     * Set additional fields that will be used in events
     * Fields needs to be passed as string in format: "field1:value1,field:value2,..."
     *
     * @param additionalFields fields to be added to event
     */
    public void setAdditionalFields(final String additionalFields) {
        this.additionalFields = Pattern.compile(",")
                .splitAsStream(additionalFields)
                .map(s -> s.split(":"))
                .collect(Collectors.toMap(a -> a[0], a -> a.length > 1 ? a[1] : ""));
    }


    /**
     * The name of a properties file that alters what is used as the
     * {@link JSON_KEYS#ENVIRONMENT} json attribute. By default in {@link DefaultJsonDataProvider} the environment
     * is the <code>studio.env</code> system property.
     * <br>
     * Each key in the properties file either starts with <code>copy-[mode].</code> or <code>split-[mode].</code>.
     * where [mode] can be "prefix" or "suffix".
     * For "split", the log message will only be sent (split) to the alternate environment.
     * For "copy", the log message will be sent to the original and alternate environment.
     * The remainder of the key gives the suffix/prefix of a logger depending on mode. e.g. <code>com.atlassian.foo</code>.
     * The value of that key is a suffix to add to the environment attribute e.g. <code>-clean</code>.
     *
     * @param filename name of config file (on classpath)
     */
    public void setEnvironmentConfigFilename(String filename) {
        cleanLogging.setEnvironmentConfigFilename(filename);
    }

    private void writeFields(
            final LogEvent event,
            Optional<String> suffix,
            final JsonGenerator g) throws IOException {
        writeBasicFields(event, suffix, g);

        writeMessageField(event, g);
        writeThrowableFields(event, g);
        writeContextFields(event, g);
        writeLocationFields(event, g);
        writeUnicornFields(g);
        writeAdditionalFields(g);
        writeExtFields(event, suffix, g);
    }

    private void writeAdditionalFields(final JsonGenerator g) {
        additionalFields.forEach((tag, value) ->
                processField(tag, g::writeStringField, () -> value, false)
        );
    }

    protected void writeMessageField(LogEvent event, JsonGenerator g) {
        processField(JSON_KEYS.MESSAGE, g::writeStringField, () -> event.getMessage() == null ? "" : event.getMessage(), true);
    }

    private void writeBasicFields(final LogEvent event, Optional<String> suffix, final JsonGenerator g) throws IOException {
        final String env = staticData.getEnvironment();

        Date date = new Date(event.getTimestamp());

        processField(JSON_KEYS.TIMESTAMP, g::writeStringField, () -> DATE_FORMAT.format(date), true);
        processField(JSON_KEYS.LEVEL, g::writeStringField, () -> event.getLevel(), true);
        processField(JSON_KEYS.SERVICE_ID, g::writeStringField, staticData::getServiceId, true);
        processField(JSON_KEYS.PRODUCT, g::writeStringField, staticData::getProductName, true);
        processField(JSON_KEYS.HOST_NAME, g::writeStringField, dataProvider::getHostName, true);
        processField(JSON_KEYS.ENVIRONMENT, g::writeStringField, () -> suffix.map(s -> env + s).orElse(env), true);
        processField(JSON_KEYS.ENV_SUFFIX, g::writeStringField, () -> suffix.orElse(null), false);
        processField(JSON_KEYS.PROCESS_ID, g::writeNumberField, staticData::getProcessId, true);
        processField(JSON_KEYS.THREAD, g::writeStringField, event::getThreadName, true);
        processField(JSON_KEYS.LOGGER, g::writeStringField, event::getLoggerName, true);
    }

    private void writeThrowableFields(final LogEvent event, final JsonGenerator g) throws IOException {
        if (event.getThrown() == null) {
            return;
        }
        StackTraceElement[] stackTraceElements = event.getStackTraceElements();
        if (stackTraceElements == null || stackTraceElements.length == 0) {
            return;
        }

        g.writeObjectFieldStart(JSON_KEYS.ERROR);

        processField(JSON_KEYS.ERROR_MESSAGE, g::writeStringField, () -> event.getThrown().getMessage());

        g.writeArrayFieldStart(JSON_KEYS.ERROR_CLASS);

        Throwable e = event.getThrown();
        do {
            g.writeString(e.getClass().getName());
            e = e.getCause();
        } while (e != null);

        g.writeEndArray();

        if (filteringApplied) {
            StringBuffer stackTrace = new StringBuffer();
            stackTraceCompressor.filterStackTrace(stackTrace, stackTraceElements);
            processField(JSON_KEYS.STACK_TRACE, g::writeStringField, stackTrace::toString);
        } else {
            processField(JSON_KEYS.STACK_TRACE, g::writeStringField, () -> STACK_TRACE_JOINER.join(stackTraceElements));
        }

        g.writeEndObject();
    }


    private void writeContextFields(final LogEvent event, final JsonGenerator g) throws IOException {
        final JsonContextData contextData = dataProvider.getContextData(event);
        if (!contextData.isEmpty()) {
            g.writeObjectFieldStart(JSON_KEYS.CONTEXT);

            processField(JSON_KEYS.REQUEST_ID, g::writeStringField, contextData::getRequestId);
            processField(JSON_KEYS.SESSION_ID, g::writeStringField, contextData::getSessionId);
            processField(JSON_KEYS.USER_KEY, g::writeStringField, contextData::getUserKey);

            g.writeEndObject();
        }
    }

    private void writeLocationFields(final LogEvent event, final JsonGenerator g) throws IOException {
        if (!includeLocation) {
            return;
        }
        LogEvent.LocationInfo location = event.getLocationInformation();

        g.writeObjectFieldStart(JSON_KEYS.LOCATION);

        processField(JSON_KEYS.CLASS, g::writeStringField, location::getClassName);
        processField(JSON_KEYS.METHOD, g::writeStringField, location::getMethodName);
        processField(JSON_KEYS.LINE_NUMBER, g::writeStringField, location::getLineNumber);

        g.writeEndObject();
    }

    private void writeUnicornFields(final JsonGenerator g) throws IOException {
        if (staticData.getDataCenter() == null && staticData.getRack() == null) {
            return;
        }

        g.writeObjectFieldStart(JSON_KEYS.UNICORN);

        processField(JSON_KEYS.DATA_CENTER, g::writeStringField, staticData::getDataCenter);
        processField(JSON_KEYS.RACK, g::writeStringField, staticData::getRack);

        g.writeEndObject();
    }

    private void writeMap(final JsonGenerator g, final Map<String, ?> map) throws IOException {
        g.writeStartObject();
        for (Map.Entry<String, ?> entry : map.entrySet()) {
            writeMapEntry(g, entry.getKey(), entry.getValue());
        }
        g.writeEndObject();
    }

    private void writeMapEntry(final JsonGenerator g, final String key, final Object value) throws IOException {
        g.writeFieldName(key);
        writeVal(g, value);
    }

    private void writeList(final JsonGenerator g, List<?> list) throws IOException {
        g.writeStartArray();
        for (Object listElement : list) {
            writeVal(g, listElement);
        }
        g.writeEndArray();
    }

    private void writeJsonable(final JsonGenerator g, final Jsonable val) throws IOException {
        StringWriter w = new StringWriter();
        val.write(w);
        g.writeRawValue(w.toString());
    }

    private void writeVal(final JsonGenerator g, final Object val) throws IOException {
        if (val instanceof Map) {
            writeMap(g, (Map) val);
        } else if (val instanceof List) {
            writeList(g, (List) val);
        } else if (val instanceof Jsonable) {
            writeJsonable(g, (Jsonable) val);
        } else {
            g.writeObject(val);
        }
    }

    private void writeExtFields(final LogEvent event, Optional<String> suffix, final JsonGenerator g) throws IOException {
        final Map<String, Object> dataTable = dataProvider.getExtraData(event);

        final Map<String, Object> map = new TreeMap<>();

        boolean isCleanEnv = suffix.isPresent();

        for (Map.Entry<String, Object> entry : dataTable.entrySet()) {
            if (isCleanEnv && cleanLogging.isSuppressed(entry.getKey()))
                continue;

            final String[] keySequence = (entry.getKey()).split("\\.");
            final Object value = entry.getValue();

            Map<String, Object> currentMap = map;
            for (int i = 0; i < keySequence.length; i++) {
                final String key = keySequence[i];
                if (currentMap.containsKey(key)) {
                    if (i == keySequence.length - 1) // If last key
                    {
                        System.err.println("Collision in MDC for key " + Arrays.toString(keySequence));
                    } else {
                        final Object mapOrVal = currentMap.get(key);
                        if (mapOrVal instanceof Map) {
                            currentMap = (Map) mapOrVal;
                        } else {
                            System.err.println("Collision in MDC for key " + Arrays.toString(keySequence));
                            break;
                        }
                    }
                } else {
                    if (i == keySequence.length - 1) // If last key
                    {
                        currentMap.put(key, value);
                        currentMap = map;
                    } else {
                        final Map<String, Object> newMapOrVal = new TreeMap<>();
                        currentMap.put(key, newMapOrVal);
                        currentMap = newMapOrVal;
                    }
                }
            }
        }

        if (map.size() == 0) {
            return;
        }

        g.writeFieldName(JSON_KEYS.EXTRA);
        writeMap(g, map);
    }


    /**
     * This method:
     * - filters fields according suppressedFields
     * - renames fields according fieldNamesMapping
     * - prevents pushing null fields to consumer
     *
     * @param fieldName field alias should be used by consumer
     * @param consumer  value consumer
     * @param producer  value producer
     */
    private <T> void processField(String fieldName, CheckedValueConsumer<T> consumer, CheckedValueProducer<T> producer, boolean keepNulls) {
        if (!suppressedFields.contains(fieldName)) {
            try {
                T value = producer.get();
                if (keepNulls || value != null) {
                    consumer.accept(fieldName, value);
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    private <T> void processField(String fieldName, CheckedValueConsumer<T> consumer, CheckedValueProducer<T> producer) {
        processField(fieldName, consumer, producer, false);
    }

    public void initialise() {
        if (dataProvider == null) {
            dataProvider = new DefaultJsonDataProvider(); // Set default implementation
        }

        stackTraceCompressor = StackTraceCompressor.defaultBuilder(minimumLines, showEludedSummary)
                .filteredFrames(filteredFrames)
                .build();

        staticData = dataProvider.getStaticData();
    }

    /**
     * An interface that describes all information that {@link AtlassianJsonLayout} needs from a log event.
     */
    public interface LogEvent {

        String getLoggerName();

        String getMessage();

        long getTimestamp();

        String getLevel();

        String getThreadName();

        Throwable getThrown();

        StackTraceElement[] getStackTraceElements();

        ReadOnlyStringMap getThreadContextMap();

        LocationInfo getLocationInformation();

        <T> T getNativeLogEvent();

        interface LocationInfo {
            String getClassName();

            String getMethodName();

            String getLineNumber();
        }
    }

}
