package com.atlassian.logging.log4j.appender.fluentd;

import javax.annotation.concurrent.ThreadSafe;
import java.util.LinkedList;
import java.util.List;


/**
 * A queue of events to send to fluentD.  Every batchPeriodMs, the contents will be sent to fluentd.
 * <p>
 * We do not use ConcurrentLinkedQueue because we already have (and need) synchronized blocks around all the access
 * points to the queue. First, using ConcurrentLinkedQueue would be a redundant level of synchronization.  Furthermore,
 * when the queue is rotated in the sender for sendoff, we need to guarantee that no threads are still enqueuing onto it
 * so that we don't lose logs.  ConcurrentLinkedQueue is "wait-free", therefore it would return before the elements are
 * necessarily added to the queue causing us to lose that guarantee.
 */
@ThreadSafe
public class LoggingEventQueue<T> {
    private final long maxNumEvents;

    volatile LinkedList<T> loggingEventQueue = new LinkedList<>();

    public LoggingEventQueue(final long maxNumEvents) {
        this.maxNumEvents = maxNumEvents;
    }

    /**
     * Add a log event to the buffer.
     * Upon completion, if buffer limit is exceeded then old events are dropped until we have some space
     *
     * @param loggingEvent the event logging to enqueue.
     */
    public synchronized void enqueue(final T loggingEvent) {
        loggingEventQueue.add(loggingEvent);

        while (loggingEventQueue.size() > maxNumEvents) {
            dequeue();
        }
    }

    /**
     * Pop up to <code>maximum</code> entries out of the queue
     *
     * @param maximum maximum number of logging entries to retrieve. Pass a negative number to retrieve all
     * @return List of logging events
     */
    public synchronized List<T> retrieveLoggingEvents(final int maximum) {
        final List<T> events = new LinkedList<>();
        if (maximum < 0 || maximum >= loggingEventQueue.size()) {
            // Just copy the queue over then reset it
            final List<T> currentQueue = loggingEventQueue;
            loggingEventQueue = new LinkedList<>();
            return currentQueue;
        } else {
            int remaining = maximum;
            while (remaining > 0) {
                events.add(dequeue());
                remaining--;
            }
        }
        return events;
    }

    /**
     * Remove and return the first log event
     *
     * @return The first log event in the buffer
     */
    public synchronized T dequeue() {
        return loggingEventQueue.poll();
    }

    public int getSize() {
        return loggingEventQueue.size();
    }

    public boolean isFull() {
        return getSize() >= maxNumEvents;
    }
}
