package com.atlassian.logging.log4j.appender.fluentd;

import java.io.Serializable;
import java.util.Timer;
import java.util.function.Function;

/**
 * Pushes logs to a FluentD endpoint.  To enable, an enabling system property must be set: -Datlassian.logging.cloud.enabled=true
 * <p>
 * Logging calls are only blocked on adding the LoggingEvent to a synchronized LinkedList (very fast).
 * <p>
 * Logs are then sent to fluentD in a batch every {@link #batchPeriodMs} ms.
 * <p>
 * With a small enough value for {@link #batchPeriodMs}, this can be near real-time.
 * <p>
 * This appender expects JSON input and will not work otherwise.
 * <p>
 * Design wiki: https://pug.jira-dev.com/wiki/display/CP/FluentD+Log+Shipper
 */
public class FluentdAppenderHelper<T> {
    private static final int DEFAULT_BATCH_PERIOD_MS = 1000;
    public static final int DEFAULT_MAX_NUM_LOG_EVENTS = 2000; // 2k log events * 1.5KB per event = 3MB
    private static final int DEFAULT_MAX_RETRY_PERIOD_MS = 1000 * 60 * 60 * 4;
    private static final int DEFAULT_BACKOFF_MULTIPLIER = 10;
    private static final int DEFAULT_MAX_BACKOFF_MINUTES = 10;

    private static final String SYS_PROP_ENABLE_KEY = "atlassian.logging.cloud.enabled";
    private static final String SYS_PROP_STD_OUT_MODE = "atlassian.logging.cloud.stdOutMode";

    /**
     * The appender will send the logs to fluentd in a batch at every batchPeriodMs milliseconds
     */
    private long batchPeriodMs = DEFAULT_BATCH_PERIOD_MS;

    /**
     * The max length of the sum of all logs in the queue before we start dropping them. This will prevent OOMEs.
     * The amount of space in bytes is roughly to maxChars * 2
     */
    private long maxNumLogEvents = DEFAULT_MAX_NUM_LOG_EVENTS;

    /**
     * The maximum amount of time that the (async) log sender will retry for before giving up.
     */
    private int maxRetryPeriodMs = DEFAULT_MAX_RETRY_PERIOD_MS;

    /**
     * The backoff multiplier Backoff formula is: `backoffMultiplier * 2^x + up to 20% jitter` where x is the attempt
     * number.
     */
    private int backoffMultiplier = DEFAULT_BACKOFF_MULTIPLIER;

    /**
     * The maximum amount of backoff for a given retry (not including 20% jitter)
     */
    private int maxBackoffMinutes = DEFAULT_MAX_BACKOFF_MINUTES;

    /**
     * The fluentd endpoint to send the logs to - including the tag.  Ex http://hostname:8888/test.tag.here
     */
    private String fluentdEndpoint;

    private LoggingEventQueue<T> loggingEventQueue;
    private FluentdLogQueueSendTask<T> sendTask;

    private volatile boolean isEnabled = false;

    private Function<T, Serializable> layout;

    /**
     * Stops and restarts the appender, picking up any configuration options that have been changed since the appender
     * was last started. If the appender has not first been enabled, then this will do nothing.
     */
    public void restart() {
        close();
        initialise();
    }

    /**
     * Enables the appender so that it can be started.
     */
    public void enable() {
        System.setProperty(SYS_PROP_ENABLE_KEY, "true");
    }

    /**
     * Disables the appender.
     */
    public void disable() {
        System.clearProperty(SYS_PROP_ENABLE_KEY);
    }

    public void initialise() {
        if (Boolean.getBoolean(SYS_PROP_ENABLE_KEY)) {
            loggingEventQueue = new LoggingEventQueue<>(maxNumLogEvents);

            final FluentdSender fluentdSender = Boolean.getBoolean(SYS_PROP_STD_OUT_MODE)
                    ? new FluentdStdOutSender()
                    : new FluentdHttpSender(fluentdEndpoint);

            sendTask = new FluentdLogQueueSendTask<>(
                    layout,
                    loggingEventQueue,
                    fluentdSender,
                    maxRetryPeriodMs,
                    backoffMultiplier,
                    maxBackoffMinutes
            );
            isEnabled = true;

            new Timer().schedule(sendTask, 0, batchPeriodMs);

        } else {
            isEnabled = false;
        }
    }

    public void append(final T loggingEvent) {
        if (isEnabled) {
            loggingEventQueue.enqueue(loggingEvent);
        }
    }

    public void close() {
        if (!isEnabled) {
            return;
        }
        isEnabled = false;

        sendTask.cancel();
        sendTask.clean();

        loggingEventQueue = null;
        sendTask = null;
    }

    public boolean isEnabled() {
        return isEnabled;
    }

    public void setFluentdEndpoint(final String fluentdEndpoint) {
        this.fluentdEndpoint = fluentdEndpoint;
    }

    public void setBatchPeriodMs(final long batchPeriodMs) {
        this.batchPeriodMs = batchPeriodMs;
    }

    public void setMaxNumEvents(final long maxNumLogEvents) {
        this.maxNumLogEvents = maxNumLogEvents;
    }

    public void setMaxRetryPeriodMs(final int maxRetryPeriodMs) {
        this.maxRetryPeriodMs = maxRetryPeriodMs;
    }

    public void setBackoffMultiplier(final int backoffMultiplier) {
        this.backoffMultiplier = backoffMultiplier;
    }

    public void setMaxBackoffMinutes(final int maxBackoffMinutes) {
        this.maxBackoffMinutes = maxBackoffMinutes;
    }

    public void setLayout(Function<T, Serializable> layout) {
        this.layout = layout;
    }
}
