package com.atlassian.logging.log4j.util;

import com.atlassian.logging.log4j.layout.json.DefaultJsonDataProvider;
import com.atlassian.logging.log4j.layout.json.JsonStaticData;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.LoggingEvent;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

public class CleanLogging {

    private Map<String, Function<String, String>> environmentMapForCopy = Collections.emptyMap();
    private Map<String, Function<String, String>> environmentMapForSplit = Collections.emptyMap();
    private Set<String> suppressed = Collections.emptySet();

    /**
     * The name of a properties file that alters what is used as the
     * JsonLayout.JSON_KEYS#ENVIRONMENT json attribute.
     * By default in {@link DefaultJsonDataProvider} the environment is the <code>studio.env</code> system property.
     *
     * Each key in the properties file either starts with <code>copy-[mode].</code> or <code>split-[mode].</code>.
     * where [mode] can be "prefix" or "suffix".
     * For "split", the log message will only be sent (split) to the alternate environment.
     * For "copy", the log message will be sent to the original and alternate environment.
     * The remainder of the key gives the suffix/prefix of a logger depending on mode. e.g. <code>com.atlassian.foo</code>.
     * The value of that key is a suffix to add to the environment attribute e.g. <code>-clean</code>.
     *
     * @param filename name of config file (on classpath)
     */
    public void setEnvironmentConfigFilename(String filename) {
        try (InputStream stream = getClass().getClassLoader().getResourceAsStream(filename)) {
            final String copy = "copy-";
            final String split = "split-";
            final String blacklist = "blacklist";
            Properties p = new Properties();
            p.load(stream);
            Map<String, Function<String, String>> copyConfig = new LinkedHashMap<>();
            Map<String, Function<String, String>> splitConfig = new LinkedHashMap<>();
            for (String prefix : p.stringPropertyNames()) {
                // Check for blacklist
                if (Objects.equals(prefix, blacklist)) {
                    suppressed = Arrays.stream(p.getProperty(blacklist).split(",")).map(String::trim).collect(Collectors.toSet());
                    continue;
                }
                // Check for env
                boolean copyEnv = prefix.startsWith(copy) && updateConfig(prefix.substring(copy.length()), p.getProperty(prefix), copyConfig);
                boolean splitEnv = prefix.startsWith(split) && updateConfig(prefix.substring(split.length()), p.getProperty(prefix), splitConfig);
                if (!copyEnv && !splitEnv) {
                    LogLog.warn("Did not understand property in : " + filename + " : " + prefix);
                }
            }
            environmentMapForCopy = Collections.unmodifiableMap(copyConfig);
            environmentMapForSplit = Collections.unmodifiableMap(splitConfig);
            } catch (IOException e) {
                LogLog.error("Unable to read env overrides from input stream : " + filename, e);
            }
    }

    /**
     * @return an alternate environment name to split and send the message to, or null if there is no alternative
     */
    public String getAlternateEnvironmentForSplit(final JsonStaticData staticData, LoggingEvent event) {
        return getAlternateEnvironment(staticData, event, environmentMapForSplit);
    }

    /**
     * @return an alternate environment name to copy and send the message to, or null if there is no alternative
     */
    public String getAlternateEnvironmentForCopy(final JsonStaticData staticData, LoggingEvent event) {
        return getAlternateEnvironment(staticData, event, environmentMapForCopy);
    }

    public boolean isSuppressed(String property) {
        return suppressed.contains(property);
    }

    private String getAlternateEnvironment(final JsonStaticData staticData, LoggingEvent event,
                                           Map<String, Function<String, String>> mapping) {
        return mapping.values().stream()
                .map(f -> f.apply(event.getLoggerName()))
                .filter(Objects::nonNull)
                .findFirst()
                .map(s -> staticData.getEnvironment() + s)
                .orElse(null);
    }

    private boolean updateConfig(String key, String value, Map<String, Function<String, String>> config) {
        String prefix = "prefix";
        String suffix = "suffix";

        int dot = key.indexOf('.');
        String mode = key.substring(0, dot);
        if (prefix.equals(mode) || suffix.equals(mode)) {
            config.put(key.substring(dot + 1), matcher(key.substring(dot + 1), value, prefix.equals(mode)));
            return true;
        }
        return false;
    }

    private Function<String, String> matcher(String pattern, String value, boolean prefix) {
        if (prefix) {
            return key -> (key.startsWith(pattern)) ? value : null;
        } else {
            return key -> (key.endsWith(pattern)) ? value : null;
        }
    }
}
