package com.atlassian.logging.log4j.juli;

import org.apache.log4j.Level;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;
import javax.annotation.Nonnull;

import static org.apache.commons.lang3.StringUtils.defaultString;

/**
 * A mapper of JULI {@link java.util.logging.LogRecord}'s
 */
public class JuliToLog4jMapper
{
    /**
     * The result of a mapping operation from JULI format
     */
    public static class MappedLogRecord
    {
        private final String callerFQCN;
        private final String loggerName;
        private final Level level;
        private final String message;
        private final Throwable throwable;

        public MappedLogRecord(final String callerFQCN, final String loggerName, final Level level, final String message, final Throwable throwable)
        {
            this.callerFQCN = callerFQCN;
            this.loggerName = loggerName;
            this.level = level;
            this.message = message;
            this.throwable = throwable;
        }

        public String getCallerFQCN()
        {
            return callerFQCN;
        }

        public Level getLevel()
        {
            return level;
        }

        public String getLoggerName()
        {
            return loggerName;
        }

        public String getMessage()
        {
            return message;
        }

        public Throwable getThrowable()
        {
            return throwable;
        }
    }

    /**
     * Maps the log record
     * @param lr the log record in plat
     * @return the mapped log record result
     */
    public MappedLogRecord map(LogRecord lr)
    {
        return new MappedLogRecord(
                mapCalledFQCN(lr),
                mapLoggerName(lr),
                mapLevel(lr),
                mapMessage(lr),
                mapThrowable(lr));
    }


    /**
     * Approximations taken (nearly) from http://logging.apache.org/log4j/2.0/log4j-jul/index.html
     */
    private static Map<java.util.logging.Level, Level> levelMap = new HashMap<>();

    static
    {
        levelMap.put(java.util.logging.Level.ALL, Level.ALL);
        levelMap.put(java.util.logging.Level.FINEST, Level.TRACE);
        levelMap.put(java.util.logging.Level.FINER, Level.DEBUG);
        levelMap.put(java.util.logging.Level.FINE, Level.DEBUG);
        levelMap.put(java.util.logging.Level.INFO, Level.INFO);
        levelMap.put(java.util.logging.Level.CONFIG, Level.INFO);
        levelMap.put(java.util.logging.Level.WARNING, Level.WARN);
        levelMap.put(java.util.logging.Level.SEVERE, Level.ERROR);
        levelMap.put(java.util.logging.Level.OFF, Level.OFF);
    }

    /**
     * The mappings sued are taken from (nearly) from http://logging.apache.org/log4j/2.0/log4j-jul/index.html.  If you
     * don't like them override this method
     *
     * @param lr the JULI log record
     * @return a log4j level
     */
    protected Level mapLevel(@Nonnull LogRecord lr)
    {
        Level lvl = levelMap.get(lr.getLevel());
        return lvl == null ? Level.ERROR : lvl;

    }

    private final SimpleFormatter formatter = new SimpleFormatter();
    /**
     * We don't expect you to override the message but you can.
     *
     * @param lr the JULI log record
     * @return a mapped message
     */
    protected String mapMessage(@Nonnull LogRecord lr)
    {
        return formatter.formatMessage(lr);
    }

    /**
     * We don't expect you to override the logger name but you can.
     *
     * @param lr the JULI log record
     * @return a mapped logger name
     */
    protected String mapLoggerName(@Nonnull LogRecord lr)
    {
        return lr.getLoggerName();
    }

    /**
     * We don't expect you to override the this but you can
     *
     * @param lr the JULI log record
     * @return the fully qualified caller class name
     */
    protected String mapCalledFQCN(@Nonnull LogRecord lr)
    {
        return defaultString(lr.getSourceClassName());
    }

    /**
     * We don't expect you to override the this but you can
     *
     * @param lr the JULI log record
     * @return the Throwable in play
     */
    protected Throwable mapThrowable(@Nonnull  LogRecord lr)
    {
        return lr.getThrown();
    }
}
