package com.atlassian.logging.log4j.appender;

import java.util.Timer;

import com.atlassian.logging.log4j.appender.fluentd.FluentdHttpSender;
import com.atlassian.logging.log4j.appender.fluentd.FluentdLogQueueSendTask;
import com.atlassian.logging.log4j.appender.fluentd.FluentdSender;
import com.atlassian.logging.log4j.appender.fluentd.FluentdStdOutSender;
import com.atlassian.logging.log4j.appender.fluentd.LoggingEventQueue;

import com.google.common.base.Preconditions;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.spi.LoggingEvent;

/**
 * Pushes logs to a FluentD endpoint.  To enable, an enabling system property must be set: -Datlassian.logging.cloud.enabled=true
 *
 * This appender is async appender by nature (benchmarks show that wrapping this in AsyncAppender only slows it down).
 * Logging calls are only blocked on adding the LoggingEvent to a synchronized LinkedList (very fast).
 *
 * Logs are then sent to fluentD in a batch every {@link #batchPeriodMs} ms.
 *
 * With a small enough value for {@link #batchPeriodMs}, this can be near real-time.
 *
 * This appender expects JSON input and will not work otherwise.
 *
 * Design wiki: https://pug.jira-dev.com/wiki/display/CP/FluentD+Log+Shipper
 */
public class FluentdAppender extends AppenderSkeleton
{
    private static final int DEFAULT_BATCH_PERIOD_MS = 1000;
    public static final int DEFAULT_MAX_NUM_LOG_EVENTS = 2000; // 2k log events * 1.5KB per event = 3MB
    private static final int DEFAULT_MAX_RETRY_PERIOD_MS = 1000 * 60 * 60 * 4;
    private static final int DEFAULT_BACKOFF_MULTIPLIER = 10;
    private static final int DEFAULT_MAX_BACKOFF_MINUTES = 10;

    private static final String SYS_PROP_ENABLE_KEY = "atlassian.logging.cloud.enabled";
    private static final String SYS_PROP_STD_OUT_MODE = "atlassian.logging.cloud.stdOutMode";

    /**
     * The appender will send the logs to fluentd in a batch at every batchPeriodMs milliseconds
     */
    private long batchPeriodMs = DEFAULT_BATCH_PERIOD_MS;

    /**
     * The max length of the sum of all logs in the queue before we start dropping them. This will prevent OOMEs.
     * The amount of space in bytes is roughly to maxChars * 2
     */
    private long maxNumLogEvents = DEFAULT_MAX_NUM_LOG_EVENTS;

    /**
     * The maximum amount of time that the (async) log sender will retry for before giving up.
     */
    private int maxRetryPeriodMs = DEFAULT_MAX_RETRY_PERIOD_MS;

    /**
     * The backoff multiplier Backoff formula is: `backoffMultiplier * 2^x + up to 20% jitter` where x is the attempt
     * number.
     */
    private int backoffMultiplier = DEFAULT_BACKOFF_MULTIPLIER;

    /**
     * The maximum amount of backoff for a given retry (not including 20% jitter)
     */
    private int maxBackoffMinutes = DEFAULT_MAX_BACKOFF_MINUTES;

    /**
     * The fluentd endpoint to send the logs to - including the tag.  Ex http://hostname:8888/test.tag.here
     */
    private String fluentdEndpoint;

    private LoggingEventQueue loggingEventQueue;
    private FluentdLogQueueSendTask sendTask;


    private final boolean isEnabled;

    public FluentdAppender() {
        isEnabled = Boolean.getBoolean(SYS_PROP_ENABLE_KEY);
    }

    protected FluentdAppender(boolean isActive) {
        this();
    }

    @Override
    public void activateOptions()
    {
        if (!isEnabled)
        {
            return;
        }

        Preconditions.checkNotNull(fluentdEndpoint);
        loggingEventQueue = new LoggingEventQueue(maxNumLogEvents);

        final FluentdSender fluentdSender =  Boolean.getBoolean(SYS_PROP_STD_OUT_MODE)
                ? new FluentdStdOutSender()
                : new FluentdHttpSender(fluentdEndpoint);

        sendTask = new FluentdLogQueueSendTask(
                getLayout(),
                loggingEventQueue,
                fluentdSender,
                maxRetryPeriodMs,
                backoffMultiplier,
                maxBackoffMinutes
        );

        new Timer().schedule(sendTask, 0, batchPeriodMs);
    }

    @Override
    protected void append(final LoggingEvent loggingEvent)
    {
        if (isEnabled)
        {
            // Read the following attributes, because they are only populated on first read, and so we need to read them
            // before handing the event to the background thread. Typically this doesn't matter as the event is logged
            // elsewhere first, but it matters if the events are sent just to this appender.
            loggingEvent.getThreadName();
            loggingEvent.getMDCCopy();
            loggingEvent.getLocationInformation();

            loggingEventQueue.enqueue(loggingEvent);
        }
    }

    @Override
    public void close()
    {
        if (!isEnabled)
        {
            return;
        }
        sendTask.cancel();
        sendTask.clean();
    }

    @Override
    public boolean requiresLayout()
    {
        return true;
    }

    public void setFluentdEndpoint(final String fluentdEndpoint)
    {
        this.fluentdEndpoint = fluentdEndpoint;
    }

    public void setBatchPeriodMs(final long batchPeriodMs)
    {
        this.batchPeriodMs = batchPeriodMs;
    }

    public void setMaxNumEvents(final long maxNumLogEvents)
    {
        this.maxNumLogEvents = maxNumLogEvents;
    }

    public void setMaxRetryPeriodMs(final int maxRetryPeriodMs)
    {
        this.maxRetryPeriodMs = maxRetryPeriodMs;
    }

    public void setBackoffMultiplier(final int backoffMultiplier)
    {
        this.backoffMultiplier = backoffMultiplier;
    }

    public void setMaxBackoffMinutes(final int maxBackoffMinutes)
    {
        this.maxBackoffMinutes = maxBackoffMinutes;
    }
}
