package com.atlassian.logging.log4j;

import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.atlassian.logging.log4j.NewLineSupport.NL;

/*
 * openutils for Log4j (http://www.openmindlab.com/lab/products/openutilslog4j.html) Copyright(C) null-2011, Openmind
 * S.r.l. http://www.openmindonline.it
 * <br>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <br>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <br>
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */

/**
 * This is based off the original code at
 * <br>
 * https://openutils.svn.sourceforge.net/svnroot/openutils/trunk/openutils-log4j/src/main/java/it/openutils/log4j/FilteredPatternLayout.java
 * <br>
 * And hence the copyright repetition. Its deviated a fair but the idea was theirs.
 * <br>
 * To use this code you need to instantiate it by using a
 * {@link com.atlassian.logging.log4j.StackTraceCompressor#defaultBuilder(int, boolean) defaultBuilder}
 * then to name the classes you wanted filtered out via
 * {@link com.atlassian.logging.log4j.StackTraceCompressor.Builder#filteredFrames(java.lang.String) filteredFrames}
 * <br>
 */
public class StackTraceCompressor {

    /**
     * Something like the following :
     * at com.atlassian.logging.log4j.StackTrace$Ponting.theBowlers(StackTrace.java:101) [test-classes/:?]
     */
    private static final Pattern CALL_SITE = Pattern.compile("(\\(.*:[0-9]+\\)) \\[.*\\]");

    /**
     * How many lines to always show
     */
    private final int minimumLines;

    /**
     * Whether to show a summary of eluded lines
     */
    private final boolean showEludedSummary;

    /**
     * Holds the list of filtered frames.
     */
    private final Set<String> filteredFrames;

    /**
     * Holds the list of everyThingAfter filtered frames.
     */
    private final Set<String> filterEveryThingAfterFrames;

    /**
     * The message to output if we filter everything after that point
     */
    private final  String filterEveryThingAfterMessage;

    /**
     * Holds the list of frames that will generate a marker
     */
    private final Set<String> markerAtFrames;

    /**
     * The message to mark specific lines at
     */
    private final String markerAtMessage;

    /**
     * The replacement string if a filter is applied to a stack trace line.
     */
    private final String filterReplacementToken;

    private StackTraceCompressor(Builder builder)
    {
        this.minimumLines = builder.minimumLines;
        this.showEludedSummary = builder.showEludedSummary;

        // Optional parameters
        this.filteredFrames = parseFrames(builder.filteredFrames);
        this.filterEveryThingAfterFrames = parseFrames(builder.filterEveryThingAfterFrames);
        this.filterEveryThingAfterMessage = StringUtils.defaultIfBlank(builder.filterEveryThingAfterMessage,
                "\t\t(The rest of the stack trace has been filtered ...)");
        this.markerAtFrames = parseFrames(builder.markerAtFrames);
        this.markerAtMessage = StringUtils.defaultIfBlank(builder.markerAtMessage, NL);
        this.filterReplacementToken = StringUtils.defaultIfBlank(builder.filterReplacementToken, "... ");
    }

    private Set<String> parseFrames(String frameSpec)
    {
        return Collections.unmodifiableSet(new SplitValueParser(",", "at ").parse(frameSpec));
    }

    public static Builder defaultBuilder(int minimumLines, boolean showEludedSummary)
    {
        return new Builder(minimumLines, showEludedSummary);
    }

    public int getMinimumLines() {
        return minimumLines;
    }

    public boolean isShowEludedSummary() {
        return showEludedSummary;
    }

    public Set<String> getFilteredFrames() {
        return filteredFrames;
    }

    public Set<String> getFilterEveryThingAfterFrames() {
        return filterEveryThingAfterFrames;
    }

    public String getFilterEveryThingAfterMessage() {
        return filterEveryThingAfterMessage;
    }

    public Set<String> getMarkerAtFrames() {
        return markerAtFrames;
    }

    public String getMarkerAtMessage() {
        return markerAtMessage;
    }

    public String getFilterReplacementToken() {
        return filterReplacementToken;
    }

    /**
     * Filter unnecessary info from stacktrace and print the result out
     * @param buffer A buffer to hold the result
     * @param stackTraceLines Array of stacktrace lines
     */
    public void filterStackTrace(StringBuffer buffer, String[] stackTraceLines)
    {
        int lineCount = 0;
        int filteredCount = 0;
        boolean ignoreLinesUntilEnd = false;
        boolean markerDue = false;
        List<String> eludedLineSummary = new ArrayList<>(stackTraceLines.length);

        // show the first N lines
        lineCount = outputMinimumLines(buffer, stackTraceLines, lineCount, getMinimumLines());
        // now filter the rest
        for (; lineCount < stackTraceLines.length; lineCount++)
        {
            String stackTraceLine = stackTraceLines[lineCount];

            boolean filteredLine = false;
            if (causedBy(stackTraceLine))
            {
                // after a cause by line we reset and show the next min lines and then start filtering all over again
                appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
                lineCount = outputMinimumLines(buffer, stackTraceLines, lineCount, getMinimumLines());
                ignoreLinesUntilEnd = false;
                continue;
            }
            else if (lineMatchesPattern(stackTraceLine, getFilterEveryThingAfterFrames()))
            {
                appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
                buffer.append(getFilterEveryThingAfterMessage());
                filteredCount = 0;
                ignoreLinesUntilEnd = true;
            }
            else if (lineMatchesPattern(stackTraceLine, getFilteredFrames()))
            {
                filteredCount++;
                filteredLine = true;
            }

            if (ignoreLinesUntilEnd)
            {
                filteredLine = true;
            }

            if (!filteredLine)
            {
                appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
                if (markerDue)
                {
                    buffer.append(getMarkerAtMessage()).append(NL);
                    markerDue = false;
                }
                buffer.append(stackTraceLine);
                if (lineMatchesPattern(stackTraceLine, getMarkerAtFrames()))
                {
                    markerDue = true;
                }
                filteredCount = 0;
            }
            else
            {
                if (isShowEludedSummary())
                {
                    String summary = makeEludedSummary(stackTraceLine);
                    if (StringUtils.isNotBlank(summary))
                    {
                        eludedLineSummary.add(summary);
                    }
                }
            }
        }
        appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
    }

    private int outputMinimumLines(StringBuffer buffer, String[] stackTraceLines, int lineCount, final int minimumLines)
    {
        int minLines = Math.min(lineCount + minimumLines, stackTraceLines.length);
        for (; lineCount < minLines; lineCount++)
        {
            String stackTraceLine = stackTraceLines[lineCount];
            buffer.append(stackTraceLine);
            if (lineCount < minLines - 1)
            {
                buffer.append(NL);
            }
        }
        return lineCount;
    }

    private void appendSkipIndicators(StringBuffer buffer, int filteredCount, List<String> eludedLineSummary)
    {
        if (filteredCount > 0)
        {
            buffer.append(NL).append("\t").append(getFilterReplacementToken())
                    .append(filteredCount).append(" filtered");
        }
        if (isShowEludedSummary())
        {
            for (String summary : eludedLineSummary)
            {
                buffer.append(" ").append(summary);
            }
            eludedLineSummary.clear();
        }
        // if it does already have a newline on the end put one there
        int lastChar = buffer.length() - 1;
        if (buffer.lastIndexOf(NL) != lastChar)
        {
            buffer.append(NL);
        }
    }

    /**
     * Check if the given string starts with any of the filtered patterns.
     *
     * @param string    checked String
     * @param filterSet the set to cechk against
     * @return <code>true</code> if the begininning of the string matches a filtered pattern, <code>false</code>
     *         otherwise
     */
    private boolean lineMatchesPattern(String string, final Set<String> filterSet)
    {
        if (!filterSet.isEmpty())
        {
            for (final String aFilterSet : filterSet)
            {
                if (string.trim().startsWith(aFilterSet))
                {
                    return true;
                }
            }
        }
        return false;
    }

    protected boolean causedBy(String stackTraceLine)
    {
        return stackTraceLine.startsWith("Caused by:");
    }

    private static String makeEludedSummary(String stackTraceLine)
    {
        Matcher matcher = CALL_SITE.matcher(stackTraceLine);
        if (matcher.find() && matcher.groupCount() >= 1) {
            return matcher.group(1);
        }
        return null;
    }

    public static class Builder {
        private int minimumLines;
        private boolean showEludedSummary;

        private String filteredFrames;
        private String filterEveryThingAfterFrames;
        private String filterEveryThingAfterMessage;
        private String markerAtFrames;
        private String markerAtMessage;
        private String filterReplacementToken;

        public Builder(int minimumLines, boolean showEludedSummary)
        {
            this.minimumLines = minimumLines;
            this.showEludedSummary = showEludedSummary;
        }

        public StackTraceCompressor build()
        {
            return new StackTraceCompressor(this);
        }

        /**
         * Set the minimum number of stack trace lines
         * @param minimumLines minimum number of stack trace lines
         * @return The builder itself
         */
        public Builder minimumLines(int minimumLines)
        {
            this.minimumLines = minimumLines;
            return this;
        }

        /**
         * Whether or not include the summary of eluded stacktrace
         * @param showEludedSummary if true, then summary is included
         * @return The builder itself
         */
        public Builder showEludedSummary(boolean showEludedSummary)
        {
            this.showEludedSummary = showEludedSummary;
            return this;
        }

        /**
         *
         * Set the list of class/package names to be filtered out from stacktrace
         * @param frameSpec Names of classes/packages to be filtered out. Can accept 3 kinds:
         *                  <br>
         *                  1. Single name (com.example)
         *                  <br>
         *                  2. Comma separated names (com.example.a,com.example.b)
         *                  <br>
         *                  3. Name of the property file which contains the actual names ({@literal @}filter_packages.properties),
         *                  see {@link com.atlassian.logging.log4j.SplitValueParser SplitValueParser}
         * @return The builder itself
         */
        public Builder filteredFrames(String frameSpec)
        {
            this.filteredFrames = frameSpec;
            return this;
        }

        public Builder filteredEveryThingAfterFrames(String frameSpec)
        {
            this.filterEveryThingAfterFrames = frameSpec;
            return this;
        }

        public Builder filteredEveryThingAfterMessage(String filterEveryThingAfterMessage)
        {
            this.filterEveryThingAfterMessage = filterEveryThingAfterMessage;
            return this;
        }

        public Builder markerAtFrames(String frameSpec)
        {
            this.markerAtFrames =  frameSpec;
            return this;
        }

        public Builder markerAtMessage(String markerAtMessage)
        {
            this.markerAtMessage = markerAtMessage;
            return this;
        }

        public Builder replacementToken(String replacementToken)
        {
            this.filterReplacementToken = replacementToken;
            return this;
        }
    }
}
