package com.atlassian.logging.log4j;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.helpers.LogLog;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

/**
 * This helper class can read separated values from a String  or from a referenced properties files.
 *
 * Its useful to use inside a log4.properties value so that you can have statements like
 * some.property=@some_file_of_values.properties or some.property=val1,val2,value3 interchangeably.
 *
 * For large values, you can avoid repeating the property entries by using the indirection
 * of using the same file reference.
 *
 * Why .properties file versus say a per line file?  Because it has comments and is a known format and log4j
 * loves properties file right?
 */
public class SplitValueParser
{
    private final String separator;
    private final String prefix;

    public SplitValueParser(final String separator, final String prefix)
    {
        this.separator = separator;
        this.prefix = prefix;
    }

    public Set<String> parse(String inputSpec)
    {
        Set<String> values = new HashSet<>();
        parseFilterSpec(inputSpec, values);
        return values;

    }

    private void parseFilterSpec(String inputSpec, final Set<String> values)
    {
        if (StringUtils.isNotBlank(inputSpec))
        {
            //
            // does it start with @ then its a properties file in the classpath
            if (inputSpec.startsWith("@"))
            {
                parseFilterSpecViaClassPath(inputSpec.substring(1), values);
            }
            else
            {
                parseFilterSpecStrings(inputSpec, values);
            }
        }
    }

    private void parseFilterSpecStrings(final String inputSpec, final Set<String> values)
    {
        if (StringUtils.isNotBlank(inputSpec))
        {
            String[] split = inputSpec.split(separator);
            for (String filter : split)
            {
                String trimmed = StringUtils.trim(filter);
                if (StringUtils.isNotBlank(trimmed))
                {
                    values.add(prefix + trimmed);
                }
            }
        }
    }

    private void parseFilterSpecViaClassPath(final String inputSpec, final Set<String> values)
    {
        InputStream stream = getClass().getClassLoader().getResourceAsStream(inputSpec);
        if (stream != null)
        {
            parseAsPropertiesFile(inputSpec, stream, values);
        }
        else
        {
            LogLog.error("Unable to load spec : " + inputSpec);
        }
    }

    private void parseAsPropertiesFile(final String inputSpec, final InputStream stream, final Set<String> values)
    {
        Properties p = new Properties();
        try
        {
            p.load(stream);
            for (Object key : p.keySet())
            {
                parseFilterSpecStrings(p.getProperty(key.toString()), values);
            }
        }
        catch (IOException e)
        {
            LogLog.error("Unable to read spec input stream : " + inputSpec);
        }
    }

}
