package com.atlassian.logging.log4j;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.helpers.PatternParser;
import org.apache.log4j.spi.LoggingEvent;
import org.apache.log4j.spi.ThrowableInformation;

import static com.atlassian.logging.log4j.NewLineSupport.NL;

/**
 * Log4j Layout that can filter out unnecessary stacktraces
 * To use this code you need to have this class as the PatternLayout and then to name the classes you wanted filtered out
 * <br>
 * <pre>
 *
 * log4j.appender.filelog.layout=com.atlassian.logging.log4j.FilteredPatternLayout
 * log4j.appender.filelog.layout.FilterFrames=org.apache.catalina, org.apache.coyote, org.apache.tomcat.util.net, sun.reflect
 *
 *
 * </pre>
 */
@SuppressWarnings("UnusedDeclaration")
public class FilteredPatternLayout extends PatternLayout
{


    /**
     * This is the master switch on whether filtering will be applied or not.  This also allows it to to switched on or
     * off at run time
     */
    private boolean filteringApplied = true;

    /**
     * This controls whether Debug level messages are filtered or not.  By default they are NOT.
     */
    private boolean filteringAppliedToDebugLevel = false;

    /**
     * Holds the list of filtered frames.
     */
    private String filteredFrames;

    /**
     * Holds the list of everyThingAfter filtered frames.
     */
    private String filterEveryThingAfterFrames;

    /**
     * Holds the list of frames that will generate a marker
     */
    private String markerAtFrames;

    /**
     * The message to output if we filter everything after that point
     */
    private String filterEveryThingAfterMessage = "\t\t(The rest of the stack trace has been filtered ...)";

    /**
     * The replacement string if a filter is applied to a stack trace line.  Default is "... "
     */
    private String filterReplacementToken = "... ";

    /**
     * The message to mark specific lines at
     */
    private String markerAtMessage = NL;

    /**
     * How many lines to always show
     */
    private int minimumLines = 6;

    /**
     * Whether to show a summary of eluded lines
     */
    private boolean showEludedSummary = false;

    /**
     * This is the master switch on whether stack trace package examination will be performed or not.  This also allows it to to switched on or
     * off at run time
     */
    private boolean stackTracePackagingExamined = true;

    /**
     * How much of a package name in logger names do we show
     */
    private int categoryNameCollapsePrecision = FqNameCollapser.NO_COLLAPSING_PRECISION;

    /**
     * Remove unnecessary lines from stacktrace
     */
    private StackTraceCompressor stackTraceCompressor;

    /**
     *
     */
    protected CategoryCollapsingPatternParser collapsingParser;

    /**
     * We must have a public no args constructor so that log4j can instantiate this class
     */
    public FilteredPatternLayout()
    {
    }


    /**
     * Any stack frame starting with <code>"at "</code> + <code>filter</code> will not be written to the log.
     * <br>
     * You can specify multiples frames separated by commas eg "org.apache.tomcat, org.hibernate"
     *
     * @param filterSpec a class name or package name to be filtered
     */
    public void setFilteredFrames(String filterSpec)
    {
        filteredFrames = filterSpec;
    }

    /**
     * A pass of the stack frame is done and frames after the matching one are ignored and will not be written to the
     * log
     *
     * @param filterSpec a class name or package name to be filtered
     */
    public void setFilterEveryThingAfterFrames(String filterSpec)
    {
        filterEveryThingAfterFrames = filterSpec;
    }

    /**
     * Any stack frame starting with <code>"at "</code> + <code>filter</code> will result in a marker message being
     * placed in the log
     *
     * @param filterSpec a class name or package name to cause a marker
     */
    public void setMarkerAtFrames(String filterSpec)
    {
        markerAtFrames = filterSpec;
    }

    /**
     * @return true if the stack traces of Debug level log events are filtered out
     */
    public boolean isFilteringAppliedToDebugLevel()
    {
        return filteringAppliedToDebugLevel;
    }


    /**
     * If set to true the stack traces of Debug level log events are filtered out.  By default there are NOT, the
     * rationale being that if its a DEBUG log event you want everything.
     *
     * @param filteringAppliedToDebugLevel a boolean flag
     */
    public void setFilteringAppliedToDebugLevel(boolean filteringAppliedToDebugLevel)
    {
        this.filteringAppliedToDebugLevel = filteringAppliedToDebugLevel;
    }

    /**
     * @return the token string that is inserted when a stack trace lines is filtered out
     */
    public String getFilterReplacementToken()
    {
        return filterReplacementToken;
    }

    /**
     * Sets the token string that is inserted when a stack trace lines is filtered out
     *
     * @param filterReplacementToken the token string to be inserted
     */
    public void setFilterReplacementToken(String filterReplacementToken)
    {
        this.filterReplacementToken = StringUtils.defaultString(filterReplacementToken);
    }

    /**
     * @return the message that is output when all stack trace lines after a given line are ignored
     */
    public String getFilterEveryThingAfterMessage()
    {
        return filterEveryThingAfterMessage;
    }

    /**
     * Sets the message that is output when all stack trace lines after a given line are ignored
     *
     * @param filterEveryThingAfterMessage the string to be inserted
     */
    public void setFilterEveryThingAfterMessage(String filterEveryThingAfterMessage)
    {
        this.filterEveryThingAfterMessage = StringUtils.defaultString(filterEveryThingAfterMessage);
    }

    /**
     * @return the message that is output if a marker stack trace line is matched
     */
    public String getMarkerAtMessage()
    {
        return filterEveryThingAfterMessage;
    }

    /**
     * Sets the message that is output if a marker stack trace line is matched
     *
     * @param markerAtMessage the string to be inserted
     */
    public void setMarkerAtMessage(String markerAtMessage)
    {
        this.markerAtMessage = StringUtils.defaultString(markerAtMessage);
    }

    /**
     * @return true if filtering will be applied to an event
     */
    public boolean isFilteringApplied()
    {
        return filteringApplied;
    }

    /**
     * Controls whether filtering will be applied to via this pattern
     *
     * @param filteringApplied the boolean flag
     */
    public void setFilteringApplied(boolean filteringApplied)
    {
        this.filteringApplied = filteringApplied;
    }

    /**
     * @return shows the first N lines of the stack trace
     */

    public int getMinimumLines()
    {
        return minimumLines;
    }

    /**
     * Sets how many first N lines of the stack trace are always shown
     *
     * @param minimumLines how many lines to always show
     */
    public void setMinimumLines(int minimumLines)
    {
        this.minimumLines = minimumLines;
    }


    /**
     * @return true if the eluded liens are shown in summary form
     */
    public boolean isShowEludedSummary()
    {
        return showEludedSummary;
    }

    /**
     * Allows the eluded line to be shown in horizontal summary form
     *
     * @param showEludedSummary true if they are to be shown
     */
    public void setShowEludedSummary(boolean showEludedSummary)
    {
        this.showEludedSummary = showEludedSummary;
    }

    /**
     * @return true of stack trace packaging examination will be performed
     */
    public boolean isStackTracePackagingExamined()
    {
        return stackTracePackagingExamined;
    }

    /**
     * Call to set whether stack trace packaging examination will be performed.
     */
    public void setStackTracePackagingExamined(final boolean stackTracePackagingExamined)
    {
        this.stackTracePackagingExamined = stackTracePackagingExamined;
    }

    /**
     * @return the default category name collapsing precision
     */
    public int getCategoryNameCollapsePrecision()
    {
        return categoryNameCollapsePrecision;
    }


    /**
     * This is really key to how this works.  Appenders by default output the stack trace UNLESS the pattern indicates
     * that it handles the exception by returning false here.
     *
     * @see org.apache.log4j.Layout#ignoresThrowable()
     */
    @Override
    public boolean ignoresThrowable()
    {
        return false;
    }


    @Override
    protected PatternParser createPatternParser(final String pattern)
    {
        collapsingParser = new CategoryCollapsingPatternParser(pattern, categoryNameCollapsePrecision);
        return collapsingParser;
    }

    /**
     * @see org.apache.log4j.PatternLayout#format(org.apache.log4j.spi.LoggingEvent)
     */
    @Override
    public String format(LoggingEvent event)
    {
        ThrowableInformation throwableInformation = event.getThrowableInformation();
        String superFormatted = super.format(event);
        if (throwableInformation == null)
        {
            return superFormatted;
        }
        return formatStackTrace(event, throwableInformation, superFormatted);
    }

    private String formatStackTrace(LoggingEvent event, ThrowableInformation throwableInformation, String formattedByPattern)
    {
        StringBuffer buffer = new StringBuffer(formattedByPattern);
        String[] stackTraceLines = getThrowableStrRep(throwableInformation);

        // if they have turned us off or its a debug event then go au natural
        if (!filteringApplied || (Level.DEBUG.equals(event.getLevel()) && !filteringAppliedToDebugLevel))
        {
            outputPlainThrowable(buffer, stackTraceLines);
        }
        else
        {
            stackTraceCompressor.filterStackTrace(buffer, stackTraceLines);
        }

        return buffer.toString();
    }

    /**
     * A hook point for subclasses to override how the lines for an exception get generated.
     * @param throwableInformation The {@link org.apache.log4j.spi.ThrowableInformation} that is associated with the
     * logging event
     * @return array containing each of the log lines that will be written.
     */
    protected String[] getThrowableStrRep(final ThrowableInformation throwableInformation)
    {
        return new StackTraceInfo(throwableInformation.getThrowable(), CommonConstants.DEFAULT_NEW_LINE_PREFIX, isStackTracePackagingExamined()).getThrowableStrRep();
    }

    private void outputPlainThrowable(StringBuffer buffer, String[] stackTraceLines)
    {
        NewLineSupport.join(buffer, stackTraceLines);
    }

    @Override
    public void activateOptions() {
        stackTraceCompressor = StackTraceCompressor.defaultBuilder(getMinimumLines(), isShowEludedSummary())
                .filteredFrames(filteredFrames)
                .filteredEveryThingAfterFrames(filterEveryThingAfterFrames)
                .filteredEveryThingAfterMessage(getFilterEveryThingAfterMessage())
                .markerAtFrames(markerAtFrames)
                .markerAtMessage(getMarkerAtMessage())
                .replacementToken(getFilterReplacementToken()).build();
    }
}
