package com.atlassian.logging.log4j.layout.json;

import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.LoggingEvent;

public class DefaultJsonDataProvider implements JsonDataProvider
{
    private static final long HOSTNAME_TTL = 30_000;
    private static final String DEFAULT_HOSTNAME = "unknown";

    private volatile String cachedHostName = null;
    private volatile long cachedHostNameValidTill = 0;

    public static enum MdcKey
    {
        REQUEST_ID("requestId"),
        SESSION_ID("sessionId"),
        USER_KEY("userKey");

        private final String key;

        MdcKey(final String key)
        {
            this.key = key;
        }

        public String getKey()
        {
            return key;
        }
    }

    private static enum SysPropKey
    {
        PRODUCT_NAME("STUDIO_COMPONENT_APP"),
        ENVIRONMENT("studio.env"),

        DATA_CENTER("unicorn.dc"),
        RACK("unicorn.rack"),
        SERVICE_ID("atlassian.logging.service.id");

        private final String key;

        SysPropKey(final String key)
        {
            this.key = key;
        }

        public String getKey()
        {
            return key;
        }
    }

    /**
     * This implementation gathers most static data from System properties, except the hostname and pid
     * See {@link com.atlassian.logging.log4j.layout.json.DefaultJsonDataProvider.SysPropKey} for keys.
     */
    @Override
    public JsonStaticData getStaticData()
    {
        final String jvmName = ManagementFactory.getRuntimeMXBean().getName();
        return JsonStaticData.builder()
                .setProductName(getSysProp(SysPropKey.PRODUCT_NAME))
                .setProcessId(getProcessId(jvmName))
                .setServiceId(getSysProp(SysPropKey.SERVICE_ID))
                .setEnvironment(getSysProp(SysPropKey.ENVIRONMENT))
                .setDataCenter(getSysProp(SysPropKey.DATA_CENTER))
                .setRack(getSysProp(SysPropKey.RACK))
                .build();
    }

    protected long getProcessId(String jvmName)
    {
        final int separatorIndex = jvmName.indexOf('@');
        if (separatorIndex < 1)
        {
            return 0L;
        }

        try
        {
            return Long.parseLong(jvmName.substring(0, separatorIndex));
        }
        catch (NumberFormatException e)
        {
            return 0L;
        }
    }

    private String getSysProp(SysPropKey key)
    {
        return System.getProperty(key.getKey());
    }

    /**
     * This implementation gets data from MDC, for key see {@link DefaultJsonDataProvider.MdcKey}
     */
    @Override
    public JsonContextData getContextData(final LoggingEvent event)
    {
        return JsonContextData.builder()
                .setRequestId(getMdc(event, MdcKey.REQUEST_ID))
                .setSessionId(getMdc(event, MdcKey.SESSION_ID))
                .setUserKey(getMdc(event, MdcKey.USER_KEY))
                .build();
    }

    private String getMdc(final LoggingEvent event, MdcKey key)
    {
        return (String) event.getProperties().get(key.getKey());
    }

    /**
     * This implementation pulls ext data from the MDC. However, the context fields are ignored since they are supplied
     * elsewhere.
     */
    @Override
    public Map<String, String> getExtraData(final LoggingEvent event)
    {
        final Map<String, String> properties = new HashMap<>(event.getProperties());

        for (MdcKey k : MdcKey.values())
        {
            properties.remove(k.getKey());
        }

        return properties;
    }

    @Override
    public String getHostName() {
        if (System.currentTimeMillis() > cachedHostNameValidTill)
        {
            // Record current timestamp
            final long now = cachedHostNameValidTill;

            // costly calculation here
            final String newHostName = resolveLocalHostName();

            // Verify the timestamp is still up to date, cancel updating if other threads already did so
            if (now == cachedHostNameValidTill)
            {
                cachedHostNameValidTill = System.currentTimeMillis() + HOSTNAME_TTL;
                cachedHostName = newHostName;
            }
        }
        return cachedHostName;
    }

    private String resolveLocalHostName()
    {
        try
        {
            return InetAddress.getLocalHost().getCanonicalHostName();
        }
        catch (UnknownHostException uhe)
        {
            LogLog.warn("Cannot resolve localhost", uhe);
            return DEFAULT_HOSTNAME;
        }
    }
}
