package com.atlassian.logging.log4j.appender.fluentd;

import java.util.List;
import java.util.Random;
import java.util.TimerTask;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.StopStrategy;
import com.github.rholder.retry.WaitStrategies;
import com.github.rholder.retry.WaitStrategy;

import org.apache.log4j.Layout;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.LoggingEvent;

/**
 * A TimerTask which sends off logging events to fluentd via http.
 *
 * This is a single-thread task - therefore, a task will not start until the previous sendoff task has finished.
 */
public class FluentdLogQueueSendTask extends TimerTask
{
    /**
     * The max number of logs to send to fluentd in a single http request.  The calls will be batched by this number.
     * This is needed to not exceed the maximum http request size.
     */
    private static final int FLUENTD_BATCH_SIZE = 50;

    private final Layout layout;
    private final LoggingEventQueue loggingEventQueue;

    private final int maxRetryPeriodMs;
    private final int backoffMultiplier;
    private final int maxBackoffMinutes;

    private final Retryer<Void> sendoffRetryer;
    private final FluentdSender fluentdSender;

    public FluentdLogQueueSendTask(final Layout layout,
            final LoggingEventQueue loggingEventQueue,
            final FluentdSender fluentdSender,
            final int maxRetryPeriodMs,
            final int backoffMultiplier,
            final int maxBackoffMinutes)
    {
        this.layout = layout;
        this.loggingEventQueue = loggingEventQueue;
        this.fluentdSender = fluentdSender;
        this.maxRetryPeriodMs = maxRetryPeriodMs;
        this.backoffMultiplier = backoffMultiplier;
        this.maxBackoffMinutes = maxBackoffMinutes;

        sendoffRetryer = buildRetryer();
    }

    private Retryer<Void> buildRetryer()
    {
        /*
         * Stop retrying after 4 hours or when the log queue reaches configured size
         */
        final StopStrategy stopStrategy = new StopStrategy()
        {
            private final StopStrategy stopAfterDelay = StopStrategies.stopAfterDelay(maxRetryPeriodMs);

            @Override
            public boolean shouldStop(final int previousAttemptNumber, final long delaySinceFirstAttemptInMillis)
            {
                if (stopAfterDelay.shouldStop(previousAttemptNumber, delaySinceFirstAttemptInMillis))
                {
                    return true;
                }
                // If we've reached the max queue size, just drop these logs so we can try the next batch instead.
                return loggingEventQueue.isFull();
            }
        };

        /*
         * Exponential + random wait between retries.
         */
        final WaitStrategy waitStrategy = new WaitStrategy()
        {
            public static final double RANDOM_RANGE_PERCENT = 1.2; // Up do 20% higher

            private final Random RANDOM = new Random();

            private WaitStrategy exponentialWait = WaitStrategies.exponentialWait(backoffMultiplier, maxBackoffMinutes, TimeUnit.MINUTES);

            @Override
            public long computeSleepTime(final int previousAttemptNumber, final long delaySinceFirstAttemptInMillis)
            {
                final long minimum = exponentialWait.computeSleepTime(previousAttemptNumber, delaySinceFirstAttemptInMillis);
                final long maximum = (long) (minimum * RANDOM_RANGE_PERCENT);
                return minimum + Math.abs(RANDOM.nextLong()) % (maximum - minimum);
            }
        };

        return RetryerBuilder.<Void>newBuilder()
                .retryIfExceptionOfType(FluentdRetryableException.class)
                .retryIfRuntimeException()
                .withWaitStrategy(waitStrategy)
                .withStopStrategy(stopStrategy)
                .build();
    }

    public void run()
    {
        final List<LoggingEvent> eventsToSend = loggingEventQueue.retrieveLoggingEvents(FLUENTD_BATCH_SIZE);

        if (eventsToSend.isEmpty())
        {
            return;
        }
        try
        {
            final String payload = buildPayload(eventsToSend);

            sendoffRetryer.call(new Callable<Void>()
            {
                public Void call() throws FluentdRetryableException
                {
                    try
                    {
                        fluentdSender.send(payload);
                    }
                    catch (Exception e)
                    {
                        LogLog.debug("Error in attempt to send logs to FluentD", e);
                        throw e;
                    }
                    return null;
                }
            });
        }
        catch (RetryException e)
        {
            // All attempts failed.  These logs are lost forever!
            LogLog.debug("FluentD logging failed - " + eventsToSend.size() + " logs lost", e);
        }
        catch (ExecutionException e)
        {
            // Any exception the callable throws besides RuntimeException and RetryableException
            // .. This should not be even possible
            LogLog.debug("FluentD logging failed for unknown reason", e);
        }
    }

    /**
     * Attemtps to send a last batch of messages.
     * You can call this method after canceling the task.
     */
    public void clean()
    {
        // Flush queue, if there many are pending messages they may be lost;
        final List<LoggingEvent> loggingEvents = loggingEventQueue.retrieveLoggingEvents(FLUENTD_BATCH_SIZE);
        if (loggingEvents.isEmpty())
        {
            return;
        }
        if (loggingEventQueue.getSize() > 0)
        {
            LogLog.warn("There are pending log messages that will be lost");
        }
        try
        {
            final String payload = buildPayload(loggingEvents);
            fluentdSender.send(payload);
        }
        catch (Exception e)
        {
            LogLog.error("Error in attempt to send logs to FluentD", e);
        }
    }

    private String buildPayload(final List<LoggingEvent> loggingEvents)
    {
        final StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("[");
        int i = 1;
        for (LoggingEvent event : loggingEvents)
        {
            stringBuilder.append(layout.format(event));
            if (i != loggingEvents.size())
            {
                stringBuilder.append(",");
            }
            i++;
        }
        stringBuilder.append("]");
        return stringBuilder.toString();
    }
}
